package com.gradle.publish;

import java.net.URI;
import java.net.URISyntaxException;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.Set;
import java.util.regex.Pattern;

class ConfigValidator {

    protected static final Collection<String> FORBIDDEN_NAMESPACES =
        Arrays.asList("org.gradle", "com.gradleware", "com.gradle", "org.samples.greeting");

    // taken directly from http://svn.apache.org/repos/asf/maven/maven-2/branches/maven-2.2.x/maven-project/src/main/java/org/apache/maven/project/validation/DefaultModelValidator.java
    // thanks StackOverflow! http://stackoverflow.com/a/2112506
    private static final Pattern MAVEN_ID_REGEX = Pattern.compile("[A-Za-z0-9_\\-.]+");
    private static final Pattern NUMBER_REGEX = Pattern.compile(".*?[0-9].*?");
    private static final Pattern VERSION_REGEX = Pattern.compile("[a-zA-Z0-9\\-.\\[\\]:+]*");

    private final boolean skipNamespaceCheck;

    ConfigValidator(boolean skipNamespaceCheck) {
        this.skipNamespaceCheck = skipNamespaceCheck;
    }

    void validateConfig(PluginsConfigurations configuration) {
        if (configuration.plugins==null || configuration.plugins.isEmpty()) {
            throw new IllegalArgumentException("No plugins defined. Please declare at least one plugin in a 'gradlePlugin.plugins' block");
        }

        validateURLs(configuration.website, configuration.vcsUrl);

        validatePluginIdUniqueness(configuration);

        for (PluginConfiguration plugin : configuration.plugins) {
            validatePluginConfig(plugin);
        }
    }

    void validatePluginConfig(PluginConfiguration plugin) {
        validatePluginId(plugin.name, plugin.id);
        validateDisplayName(plugin.name, plugin.displayName);
        validateDescription(plugin.name, plugin.description);
        validateImplementationClass(plugin.name, plugin.implementationClass);
        validateTags(plugin);
    }

    void validateTags(PluginConfiguration plugin) {
        validateTags(plugin.name, plugin.tags);
    }

    void validateMavenCoordinates(String groupId, String artifactId, String version) {
        if (groupId==null || groupId.isEmpty()) {
            throw new IllegalArgumentException("Cannot publish artifacts as no group ID set.\n"
                + "Please set the group at the project level.");
        }
        if (!MAVEN_ID_REGEX.matcher(groupId).matches()) {
            throw new IllegalArgumentException(
                "Cannot publish artifacts as group ID '" + groupId + "' is invalid.\n"
                    + "Valid group IDs may include alphanumeric characters, numbers, dashes,\n"
                    + "underscores and dots.");
        }
        if (failsNamespaceCheck(groupId)) {
            throw new IllegalArgumentException(
                "Cannot publish artifacts as group ID '" + groupId + "' is not allowed.\n"
                    + "Group IDs cannot begin with " + String.join(", ", FORBIDDEN_NAMESPACES));
        }

        if (artifactId==null || artifactId.isEmpty()) {
            throw new IllegalArgumentException("Cannot publish artifacts as no artifact ID set.\n"
                + "Please set the artifact ID at the project level.");
        }
        if (!MAVEN_ID_REGEX.matcher(artifactId).matches()) {
            throw new IllegalArgumentException(
                "Cannot publish artifacts as artifact ID '" + artifactId + "' is invalid.\n"
                    + "Valid artifact IDs may include alphanumeric characters, numbers, dashes,\n"
                    + "underscores and dots.");
        }

        if (artifactId.equals(groupId + ".gradle.plugin")) {
            throw new IllegalArgumentException("Group ID + '.gradle.plugin' suffix not allowed as artifact ID");
        }

        validateVersion(version);
    }

    private void validateVersion(String version) {
        if (version==null || version.isEmpty() || version.equals("unspecified")) {
            throw new IllegalArgumentException("Cannot publish artifacts as no version set.\n"
                + "Please set the version at the project level.");
        }

        if (!NUMBER_REGEX.matcher(version).matches() || !VERSION_REGEX.matcher(version).matches() || version.length() > 50) {
            throw new IllegalArgumentException("Invalid version '" + version + "'. Project version string must: " +
                "1) include a number;" +
                "2) be less than 50 characters long;" +
                "3) match the regular expression: " + VERSION_REGEX.pattern());
        }
        if (version.trim().endsWith("-SNAPSHOT")) {
            throw new IllegalArgumentException("-SNAPSHOT plugin versions not supported, please use a fixed version instead.");
        }
    }

    protected void validateURLs(String websiteUrl, String vcsUrl) {
        validateURL(websiteUrl, "Website");
        validateURL(vcsUrl, "VCS");
    }

    private void validateURL(String url, String description) {
        if (isNullOrBlank(url)) {
            throw new IllegalArgumentException(description + " URL not set");
        }
        try {
            URI normalizedURI = new URI(url.toLowerCase());
            String scheme = normalizedURI.getScheme();
            if (!scheme.equals("http") && !scheme.equals("https")) {
                throw new IllegalArgumentException(description + " URL only supports HTTP or HTTPS schemas");
            }
        } catch (URISyntaxException e) {
            throw new IllegalArgumentException("Can't parse URL '" + url + "'");
        }
    }

    protected void validatePluginIdUniqueness(PluginsConfigurations configuration) {
        Set<String> usedPluginIds = new HashSet<>();
        for (PluginConfiguration plugin : configuration.plugins) {
            String pluginId = plugin.id;
            if (usedPluginIds.contains(pluginId)) {
                throw new InvalidPluginIdException(pluginId, "multiple plugins are using the same ID");
            }
            usedPluginIds.add(pluginId);
        }
    }

    protected void validatePluginId(String pluginName, String pluginId) {
        if (pluginId==null || pluginId.isEmpty()) {
            throw new IllegalArgumentException("Plugin " + pluginName + " does not define a plugin ID. Please set the 'id' property in the plugin declaration");
        }
        PluginId id = PluginId.of(pluginId);
        if (!id.isQualified()) {
            throw new InvalidPluginIdException(pluginId, "Plugin IDs should be namespaced, e.g. 'com.example." + pluginId + "'");
        }
        if (failsNamespaceCheck(pluginId)) {
            throw new InvalidPluginIdException(pluginId, "cannot begin with " + FORBIDDEN_NAMESPACES);
        }
    }

    protected void validateTags(String pluginName, Collection<String> tags) {
        if (tags==null || tags.isEmpty()) {
            throw new IllegalArgumentException("Plugin '" + pluginName + "' has no 'tags' property set");
        }
    }

    protected void validateDisplayName(String pluginName, String displayName) {
        validateProperty(pluginName, "displayName", displayName);
    }

    protected void validateDescription(String pluginName, String description) {
        validateProperty(pluginName, "description", description);
    }

    protected void validateImplementationClass(String pluginName, String implementationClass) {
        validateProperty(pluginName, "implementationClass", implementationClass);
    }

    private void validateProperty(String pluginName, String propertyName, String propertyValue) {
        if (isNullOrBlank(propertyValue)) {
            throw new IllegalArgumentException("Plugin '" + pluginName + "' has no '" + propertyName + "' property set");
        }
    }

    private boolean failsNamespaceCheck(String id) {
        if (!skipNamespaceCheck) {
            id = id.toLowerCase();
            for (String ns : FORBIDDEN_NAMESPACES) {
                if (id.equals(ns) || id.startsWith(ns + ".")) {
                    return true;
                }
            }
        }
        return false;
    }

    private static boolean isNullOrBlank(String src) {
        return src==null || src.trim().isEmpty();
    }

    static class PluginId {

        public static final String ID_SEPARATOR_ON_START_OR_END = "cannot begin or end with '" + PluginId.SEPARATOR + "'";
        public static final String DOUBLE_SEPARATOR = "cannot contain '" + PluginId.SEPARATOR + PluginId.SEPARATOR + "'";

        public static final String PLUGIN_ID_VALID_CHARS_DESCRIPTION = "ASCII alphanumeric characters, '.', '_' and '-'";
        public static final String SEPARATOR = ".";

        private final String value;

        private PluginId(String value) {
            this.value = value;
        }

        static PluginId of(String value) throws InvalidPluginIdException {
            validate(value);
            return new PluginId(value);
        }

        static void validate(String value) throws InvalidPluginIdException {
            if (value.startsWith(SEPARATOR) || value.endsWith(SEPARATOR)) {
                throw new InvalidPluginIdException(value, ID_SEPARATOR_ON_START_OR_END);
            } else if (value.contains(PluginId.SEPARATOR + PluginId.SEPARATOR)) {
                throw new InvalidPluginIdException(value, DOUBLE_SEPARATOR);
            } else {
                Character invalidChar = invalidChar(value);
                if (invalidChar!=null) {
                    throw new InvalidPluginIdException(value, invalidPluginIdCharMessage(invalidChar));
                }
            }
        }

        static String invalidPluginIdCharMessage(char invalidChar) {
            return "contains invalid char '" + invalidChar + "' (only " + PluginId.PLUGIN_ID_VALID_CHARS_DESCRIPTION + " characters are valid)";
        }

        boolean isQualified() {
            return value.contains(PluginId.SEPARATOR);
        }

        @Override
        public String toString() {
            return value;
        }

        @Override
        public boolean equals(Object o) {
            if (this==o) {
                return true;
            }
            if (o==null || getClass()!=o.getClass()) {
                return false;
            }

            PluginId pluginId = (PluginId) o;

            return value.equals(pluginId.value);
        }

        @Override
        public int hashCode() {
            return value.hashCode();
        }

        private static Character invalidChar(String src) {
            for (char c : src.toCharArray()) {
                if (!(inRange('A', 'Z', c) || inRange('a', 'z', c) || inRange('0', '9', c)
                    || c=='.' || c=='-' || c=='_')) {
                    return c;
                }
            }
            return null;
        }

        private static boolean inRange(char start, char end, char c) {
            return c >= start && c <= end;
        }
    }
}
