package com.gradle.publish;

import org.gradle.api.Project;
import org.gradle.api.provider.Provider;
import org.gradle.api.provider.ValueSource;

import javax.annotation.Nullable;

public abstract class CredentialsValueSource implements ValueSource<Credentials, CredentialsValueParameters> {
    public static final String GRADLE_PUBLISH_SECRET = "gradle.publish.secret";
    public static final String GRADLE_PUBLISH_KEY = "gradle.publish.key";
    public static final String GRADLE_PUBLISH_SECRET_ENV = "GRADLE_PUBLISH_SECRET";
    public static final String GRADLE_PUBLISH_KEY_ENV = "GRADLE_PUBLISH_KEY";

    @Override
    public Credentials obtain() {
        CredentialsValueParameters params = getParameters();
        String key = getEnvThenSystemProperty(GRADLE_PUBLISH_KEY_ENV, params.getGradlePublishKey(), GRADLE_PUBLISH_KEY);
        String secret = getEnvThenSystemProperty(GRADLE_PUBLISH_SECRET_ENV, params.getGradlePublishSecret(), GRADLE_PUBLISH_SECRET);
        if (key==null || key.trim().isEmpty() || secret==null || secret.trim().isEmpty()) {
            throw new IllegalArgumentException("Missing publishing keys. Please set " +
                GRADLE_PUBLISH_KEY + "/" + GRADLE_PUBLISH_SECRET +
                " system properties or " + GRADLE_PUBLISH_KEY_ENV + "/" + GRADLE_PUBLISH_SECRET_ENV +
                " env variables or login using the " + PublishPlugin.LOGIN_TASK_NAME +
                " task.");
        }
        return new Credentials(key, secret);
    }

    @Nullable
    private String getEnvThenSystemProperty(String envName, Provider<String> projectPropertiesProvider, String systemPropertyName) {
        String envValue = System.getenv(envName);
        if (envValue==null) {
            return projectPropertiesProvider.getOrElse(System.getProperty(systemPropertyName));
        } else {
            return envValue;
        }
    }

    static Provider<Credentials> create(Project project) {
        return project.getProviders().of(CredentialsValueSource.class, params -> {
            params.getParameters().getGradlePublishKey().value(project.getProviders().gradleProperty(GRADLE_PUBLISH_KEY));
            params.getParameters().getGradlePublishSecret().value(project.getProviders().gradleProperty(GRADLE_PUBLISH_SECRET));
        });
    }
}
