package com.gradle.publish;

import com.gradle.publish.protocols.v1.models.login.LoginCheckRequest;
import com.gradle.publish.protocols.v1.models.login.LoginCheckResponse;
import com.gradle.publish.protocols.v1.models.login.LoginInitRequest;
import com.gradle.publish.protocols.v1.models.login.LoginInitResponse;
import org.gradle.api.DefaultTask;
import org.gradle.api.Transformer;
import org.gradle.api.file.RegularFileProperty;
import org.gradle.api.provider.Property;
import org.gradle.api.provider.Provider;
import org.gradle.api.tasks.Input;
import org.gradle.api.tasks.OutputFile;
import org.gradle.api.tasks.TaskAction;
import org.gradle.work.DisableCachingByDefault;
import org.jetbrains.annotations.NotNull;

import java.io.File;
import java.io.IOException;
import java.net.HttpURLConnection;
import java.net.InetAddress;
import java.net.URL;
import java.util.LinkedHashMap;
import java.util.Map;

import static com.gradle.publish.CredentialsValueSource.GRADLE_PUBLISH_KEY;
import static com.gradle.publish.CredentialsValueSource.GRADLE_PUBLISH_SECRET;

@DisableCachingByDefault(because = "Not cacheable")
public abstract class LoginTask extends DefaultTask {

    @OutputFile
    public abstract RegularFileProperty getGradleHomeProperties();

    @Input
    public abstract Property<String> getPluginPortalUrl();

    private final Provider<Long> loginSleepIntervalMs = getProject().getProviders()
        .systemProperty("gradle.portal.login.sleep")
        .map(new LongTransformer())
        .orElse(500L);

    private final Provider<Long> loginMaxWaitMs = getProject().getProviders()
        .systemProperty("gradle.portal.login.wait")
        .map(new LongTransformer())
        .orElse(5L * 1000 * 60);

    private boolean isLoggedIn(LoginCheckRequest loginCheckRequest) throws IOException {
        URL url = new URL(getPluginPortalUrl().get() + loginCheckRequest.requestProtocolURL());
        HttpURLConnection connection = (HttpURLConnection) url.openConnection();
        connection.setRequestMethod(loginCheckRequest.getRequestMethod());
        String response = ResponseUtil.readResponse(connection);
        getLogger().debug("Is logged in response: {}", response);
        LoginCheckResponse loginCheckResponse = loginCheckRequest.convertResponse(response);

        // NOTE: This was only added in 0.11.0
        ResponseUtil.assertValidResponse("Failed checking if user is logged in.", loginCheckResponse);

        connection.disconnect();
        if (loginCheckResponse.getCompleted()) {
            String key = loginCheckResponse.getKey();
            String secret = loginCheckResponse.getSecret();
            updateKeySecretProperties(key, secret, loginCheckResponse.getSuccessMessage());
        }
        return loginCheckResponse.getCompleted();
    }

    private void updateKeySecretProperties(String key, String secret, String message) throws IOException {
        File propertiesFile = getGradleHomeProperties().getAsFile().get();
        getLogger().debug("Updating key and secret in file: {}", propertiesFile.getAbsolutePath());
        Map<String, String> props = new LinkedHashMap<>();
        props.put(GRADLE_PUBLISH_KEY, key);
        props.put(GRADLE_PUBLISH_SECRET, secret);
        PropertiesStore.append(propertiesFile, props, "Updated secret and key with server message: " + message);
        getLogger().quiet("Your publishing API keys have been written to: " + propertiesFile.getAbsolutePath());
    }

    @TaskAction
    public void login() throws IOException, InterruptedException {
        String keyName = InetAddress.getLocalHost().getHostName();
        LoginInitRequest request = new LoginInitRequest(keyName);
        URL url = new URL(getPluginPortalUrl().get() + request.requestProtocolURL());
        HttpURLConnection connection = (HttpURLConnection) url.openConnection();
        connection.setRequestMethod(request.getRequestMethod());

        RequestUtil.addClientVersionAsRequestHeader(connection);

        String stringResponse = ResponseUtil.readResponse(connection);
        int status = connection.getResponseCode();
        getLogger().debug("Login response: {}", stringResponse);
        LoginInitResponse apiResponse = ResponseUtil.convertResponse(request, url, stringResponse, status);

        // NOTE: This was only added in 0.11.0
        ResponseUtil.assertValidResponse("Failed login initialization request.", apiResponse);

        File propertiesFile = getGradleHomeProperties().get().getAsFile();
        getLogger().quiet(
            "\nTo add your publishing API keys to {}, open the following URL in your browser:\n\n    {}\n\n",
            propertiesFile.getAbsolutePath(),
            apiResponse.getValidateUrl()
        );
        connection.disconnect();
        long startTime = System.currentTimeMillis();
        long maxWait = loginMaxWaitMs.get();
        long loginSleepInterval = loginSleepIntervalMs.get();
        while (!isLoggedIn(apiResponse.getNextRequest()) &&
            (System.currentTimeMillis() - startTime) < maxWait) {
            Thread.sleep(loginSleepInterval);
            getLogger().trace("Trying login again: {}", apiResponse.getToken());
        }
    }

    // workaround for Gradle 7.4 issue when transformer lamba is not serializable
    private static final class LongTransformer implements Transformer<Long, String> {
        @Override
        public Long transform(@NotNull String original) {
            return Long.parseLong(original);
        }
    }

}
