package com.gradle.publish;

import com.gradle.protocols.ServerResponseBase;
import com.gradle.publish.protocols.v1.models.ClientPostRequest;
import oauth.signpost.basic.DefaultOAuthConsumer;
import oauth.signpost.exception.OAuthException;
import org.gradle.api.logging.Logger;
import org.gradle.api.logging.Logging;

import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.URI;
import java.net.URL;
import java.nio.charset.StandardCharsets;

class OAuthHttpClient {

    private static final Logger LOGGER = Logging.getLogger(OAuthHttpClient.class);

    private static final String CONSUMER_KEY = "gradle";
    private static final String CONSUMER_SECRET = "gradle";

    private final URI baseUrl;
    private final String publishKey;
    private final String publishSecret;

    OAuthHttpClient(String baseUrl, String publishKey, String publishSecret) {
        this.baseUrl = URI.create(baseUrl);
        this.publishKey = publishKey;
        this.publishSecret = publishSecret;
    }

    private void sign(HttpURLConnection con) {
        DefaultOAuthConsumer consumer = new DefaultOAuthConsumer(CONSUMER_KEY, CONSUMER_SECRET);
        consumer.setTokenWithSecret(publishKey, publishSecret);
        try {
            consumer.sign(con);
        } catch (OAuthException e) {
            throw new RuntimeException("Error signing request", e);
        }
    }

    <T extends ServerResponseBase> T send(ClientPostRequest<T> postRequest)
        throws Exception {
        String requestJson = postRequest.getPostJsonString();
        // these are all small, don't bother with streams etc here
        byte[] requestBytes = requestJson.getBytes(StandardCharsets.UTF_8);

        URL url = baseUrl.resolve(postRequest.requestProtocolURL()).toURL();
        HttpURLConnection con = (HttpURLConnection) url.openConnection();
        con.setRequestMethod("POST");
        con.setDoOutput(true);
        con.addRequestProperty("Content-Type", postRequest.getContentType());
        con.addRequestProperty("Content-Length", String.valueOf(requestBytes.length));
        RequestUtil.addClientVersionAsRequestHeader(con);

        sign(con);

        LOGGER.debug("Connecting to: {}", url);
        con.connect();

        try (OutputStream os = con.getOutputStream()) {
            os.write(requestBytes);
        }

        int statusCode = con.getResponseCode();
        boolean error = statusCode >= 400 || statusCode < 200;
        String responseJson = ResponseUtil.readResponse(error ? con.getErrorStream():con.getInputStream());
        LOGGER.debug("Signed post response with code: {} was: {}", statusCode, responseJson);
        T apiResponse = ResponseUtil.convertResponse(postRequest, url.toExternalForm(), responseJson, statusCode);
        if (statusCode == 401) {
            throw new  RuntimeException("Unauthorized!" +
                "\n  Login from command line using:" +
                "\n    gradle login" +
                "\n  or login and download user API keys on:" +
                "\n    " + baseUrl + "/user/login");
        } else if (error) {
            ResponseUtil.assertValidResponse("Failed to post to server.", apiResponse);
        }

        return apiResponse;
    }

}
