package com.gradle.publish;

import org.gradle.api.GradleException;
import org.gradle.api.NamedDomainObjectContainer;
import org.gradle.api.Plugin;
import org.gradle.api.Project;
import org.gradle.api.Task;
import org.gradle.api.artifacts.ResolvableDependencies;
import org.gradle.api.artifacts.component.ComponentIdentifier;
import org.gradle.api.artifacts.component.ModuleComponentIdentifier;
import org.gradle.api.artifacts.result.ResolutionResult;
import org.gradle.api.internal.artifacts.result.DefaultResolvedDependencyResult;
import org.gradle.api.logging.Logger;
import org.gradle.api.logging.Logging;
import org.gradle.api.plugins.JavaPluginExtension;
import org.gradle.api.provider.Provider;
import org.gradle.api.publish.Publication;
import org.gradle.api.publish.PublishingExtension;
import org.gradle.language.base.plugins.LifecycleBasePlugin;
import org.gradle.plugin.devel.GradlePluginDevelopmentExtension;
import org.gradle.plugin.devel.PluginDeclaration;
import org.gradle.plugins.signing.SigningExtension;
import org.gradle.util.GradleVersion;

import javax.annotation.Nonnull;
import java.io.File;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import static org.codehaus.groovy.runtime.StringGroovyMethods.capitalize;
import static org.gradle.util.GradleVersion.version;

public class PublishPlugin implements Plugin<Project> {
    public static final String JAVA_GRADLE_PLUGIN_ID = "java-gradle-plugin";
    public static final String MAVEN_PUBLISH_PLUGIN_ID = "maven-publish";
    public static final String SHADOW_PLUGIN_ID = "com.github.johnrengelman.shadow";
    public static final String NEW_SHADOW_PLUGIN_ID = "com.gradleup.shadow";
    public static final String SIGNING_PLUGIN_ID = "signing";

    public static final String LOGIN_TASK_NAME = "login";
    private static final String PUBLISH_TASK_DESCRIPTION = "Publishes this plugin to the " +
        "Gradle Plugin portal.";
    private static final String LOGIN_TASK_DESCRIPTION = "Update the gradle.properties files " +
        "so this machine can publish to the Gradle Plugin portal.";
    private static final String PORTAL_BUILD_GROUP_NAME = "Plugin Portal";

    private static final String PUBLISH_TASK_NAME = "publishPlugins";
    public static final String PLUGIN_BUNDLE_EXTENSION_NAME = "pluginBundle";

    static final String MAVEN_PUBLICATION_NAME = "pluginMaven";
    static final String MARKER_PUBLICATION_SUFFIX = "PluginMarkerMaven";

    static final String PORTAL_URL_PROPERTY_ENV = "GRADLE_PORTAL_URL";
    static final String PORTAL_URL_PROPERTY = "gradle.portal.url";

    private static final Logger LOGGER = Logging.getLogger(PublishPlugin.class);

    @Override
    public void apply(@Nonnull final Project project) {
        if (runningOnPreGradle(version("7.4"))) {
            throw new RuntimeException("This version of the Plugin Publish Plugin isn't compatible with Gradle versions older than 7.4");
        }

        project.getPlugins().apply("org.gradle.java-library");

        Log4jVulnerabilityChecker.decorate(project);

        wireToJavaGradlePluginAndMavenPublish(project);

        Provider<String> portalUrl = project.getProviders()
            .environmentVariable(PORTAL_URL_PROPERTY_ENV)
            .orElse(project.getProviders().systemProperty(PORTAL_URL_PROPERTY))
            .orElse("https://plugins.gradle.org");

        Provider<PluginsConfigurations> pluginsConfigurations = getPluginsConfigurations(project);

        project.getTasks().register(PUBLISH_TASK_NAME, PublishTask.class, publishTask -> {
            publishTask.setDescription(PUBLISH_TASK_DESCRIPTION);
            publishTask.setGroup(PORTAL_BUILD_GROUP_NAME);

            publishTask.getConfig().set(pluginsConfigurations);
            publishTask.getPluginPortalUrl().convention(portalUrl);
        });

        LOGGER.debug("Setup: {} of {}", PUBLISH_TASK_NAME, getClass().getName());

        project.getTasks().register(LOGIN_TASK_NAME, LoginTask.class, loginTask -> {
            loginTask.setDescription(LOGIN_TASK_DESCRIPTION);
            loginTask.setGroup(PORTAL_BUILD_GROUP_NAME);

            loginTask.getGradleHomeProperties().set(new File(project.getGradle().getGradleUserHomeDir(), Project.GRADLE_PROPERTIES));
            loginTask.getPluginPortalUrl().convention(portalUrl);
        });

        LOGGER.debug("Created task: {} of {}", LOGIN_TASK_NAME, getClass().getName());

        project.afterEvaluate(finalProject -> {
            final GradlePluginDevelopmentExtension pluginDevelopmentExtension = project.getExtensions().getByType(GradlePluginDevelopmentExtension.class);
            if (!pluginDevelopmentExtension.isAutomatedPublishing()) {
                throw new RuntimeException("Since version 1.0 of the Plugin Publish plugin non-automatic publishing is not allowed anymore; the Maven Publish plugin must be used to generate publication metadata");
            }

            PublishTaskShadowAction shadowAction = new PublishTaskShadowAction(project, LOGGER);
            project.getPluginManager().withPlugin(PublishPlugin.NEW_SHADOW_PLUGIN_ID, shadowAction);
            project.getPluginManager().withPlugin(PublishPlugin.SHADOW_PLUGIN_ID, shadowAction);

            forceJavadocAndSourcesJars(project);

            List<Task> signingTasks = enableSigning(project);
            PublishTask publishTask = (PublishTask) project.getTasks().findByName(PUBLISH_TASK_NAME);
            if (publishTask!=null) {
                publishTask.dependsOn(LifecycleBasePlugin.ASSEMBLE_TASK_NAME);
                signingTasks.forEach(publishTask::dependsOn);
                publishTask.dependOnPublishTasks();
            }

            project.getTasks().withType(PublishExistingTask.class).configureEach(task ->
                task.getPluginPortalUrl().convention(portalUrl)
            );
        });
    }

    @SuppressWarnings("deprecation")
    private Provider<PluginsConfigurations> getPluginsConfigurations(Project project) {
        if (!runningOnPreGradle(version("8.0"))) {
            return project.provider(() ->
                new PluginsConfigurations(project.getExtensions().getByType(GradlePluginDevelopmentExtension.class))
            );
        }

        project.getExtensions().add(PLUGIN_BUNDLE_EXTENSION_NAME, new PluginBundleExtension());

        return project.provider(() -> {
            PluginBundleExtension bundle = project.getExtensions().getByType(PluginBundleExtension.class);
            GradlePluginDevelopmentExtension plugins = project.getExtensions().getByType(GradlePluginDevelopmentExtension.class);

            if (!bundle.isEmpty()) {
                if (!runningOnPreGradle(version("7.6-dev-1"))) {
                    LOGGER.warn(
                        "Using the `pluginBundle` block to configure plugin publication has been deprecated. " +
                            "This is scheduled to be removed in Gradle 8.0. " +
                            "Use new properties of `gradlePlugin` instead. " +
                            "See https://docs.gradle.org/8.0/userguide/publishing_gradle_plugins.html#sec:configuring-publish-plugin for more details."
                    );
                }
                validateLegacyPluginTagKeys(bundle, plugins);
                return new PluginsConfigurations(
                    bundle.getWebsite(),
                    bundle.getVcsUrl(),
                    plugins.getPlugins().stream().map(declaration -> new PluginConfiguration(
                        declaration.getName(),
                        declaration.getId(),
                        declaration.getDisplayName(),
                        declaration.getDescription() == null ? bundle.getDescription() : declaration.getDescription(),
                        declaration.getImplementationClass(),
                        new ArrayList<>(getLegacyTags(bundle, declaration.getName()))
                    )).collect(Collectors.toList())
                );
            } else {
                return new PluginsConfigurations(project.getExtensions().getByType(GradlePluginDevelopmentExtension.class));
            }
        });
    }

    @SuppressWarnings("deprecation")
    private static Collection<String> getLegacyTags(PluginBundleExtension bundleConfig, String pluginName) {
        Collection<String> tags = bundleConfig.getPluginTags().get(pluginName);
        if (tags != null && !tags.isEmpty()) {
            return tags;
        }

        return bundleConfig.getTags();
    }

    @SuppressWarnings("deprecation")
    private void validateLegacyPluginTagKeys(PluginBundleExtension bundleConfig, GradlePluginDevelopmentExtension extension) {
        Set<String> plugins = new HashSet<>(bundleConfig.getPluginTags().keySet());
        plugins.removeAll(extension.getPlugins().getNames());
        if (!plugins.isEmpty()) {
            List<String> sortedPlugins = new ArrayList<>(plugins);
            sortedPlugins.sort(String::compareTo);
            throw new IllegalArgumentException("Tags set for undefined plugin(s): " + String.join(", ", sortedPlugins));
        }
    }

    private List<Task> enableSigning(Project project) {
        List<Task> signingTasks = new ArrayList<>();
        project.getPluginManager().withPlugin(PublishPlugin.SIGNING_PLUGIN_ID, signingPlugin -> {
            LOGGER.info("Signing plugin detected. Will automatically sign the published artifacts.");

            SigningExtension signing = project.getExtensions().getByType(SigningExtension.class);
            PublishingExtension publishing = project.getExtensions().getByType(PublishingExtension.class);
            Task mavenPublishSignTask = getSigningTask(project, signing, publishing.getPublications().getByName(MAVEN_PUBLICATION_NAME));
            if (mavenPublishSignTask != null) {
                signingTasks.add(mavenPublishSignTask);
            }

            NamedDomainObjectContainer<PluginDeclaration> plugins = project.getExtensions().getByType(GradlePluginDevelopmentExtension.class).getPlugins();
            for (PluginDeclaration plugin : plugins) {
                String markerPublicationName = plugin.getName() + MARKER_PUBLICATION_SUFFIX;
                Publication publication = publishing.getPublications().getByName(markerPublicationName);
                Task pluginSigningTask = getSigningTask(project, signing, publication);
                if (pluginSigningTask != null) {
                    signingTasks.add(pluginSigningTask);
                }
            }
        });
        return signingTasks;
    }

    private Task getSigningTask(Project project, SigningExtension signing, Publication publication) {
        String signTaskName = determineSignTaskNameForPublication(publication);
        Task signTask = project.getTasks().findByName(signTaskName);
        if (signTask == null) {
            signTask = signing.sign(publication).get(0);
        }
        return signTask;
    }

    private static String determineSignTaskNameForPublication(Publication publication) {
        return "sign" + capitalize(publication.getName()) + "Publication";
    }

    private void forceJavadocAndSourcesJars(Project project) {
        JavaPluginExtension javaPluginExtension = project.getExtensions().getByType(JavaPluginExtension.class);
        javaPluginExtension.withJavadocJar();
        javaPluginExtension.withSourcesJar();
    }

    private void wireToJavaGradlePluginAndMavenPublish(final Project project) {
        project.getPluginManager().apply(JAVA_GRADLE_PLUGIN_ID);
        project.getPluginManager().apply(MAVEN_PUBLISH_PLUGIN_ID);
    }

    private static boolean runningOnPreGradle(GradleVersion version) {
        return GradleVersion.current().getBaseVersion().compareTo(version) < 0;
    }

    private static class Log4jVulnerabilityChecker {

        public static final String ERROR_MESSAGE = "Cannot publish a plugin which resolves a vulnerable Log4j version (https://blog.gradle.org/log4j-vulnerability). " +
            "Make sure to update your configuration so it does not happen.";

        static void decorate(Project project) {
            project.getConfigurations().all(conf -> {
                ResolvableDependencies incoming = conf.getIncoming();
                incoming.afterResolve(rd -> {
                    ResolutionResult resolutionResult = rd.getResolutionResult();
                    resolutionResult.allDependencies(dependencyResult -> {
                        if (dependencyResult instanceof DefaultResolvedDependencyResult) {
                            ComponentIdentifier componentIdentifier = ((DefaultResolvedDependencyResult) dependencyResult).getSelected().getId();
                            if (componentIdentifier instanceof ModuleComponentIdentifier) {
                                ModuleComponentIdentifier moduleComponentIdentifier = (ModuleComponentIdentifier) componentIdentifier;
                                if (isVulnerableLog4jDependency(moduleComponentIdentifier)) {
                                    throw new GradleException(ERROR_MESSAGE);
                                }
                            }
                        }
                    });
                });
            });
        }

        private static boolean isVulnerableLog4jDependency(ModuleComponentIdentifier identifier) {
            if (!"org.apache.logging.log4j".equals(identifier.getGroup())) {
                return false;
            }

            if (!"log4j-core".equals(identifier.getModule())) {
                return false;
            }

            String[] versionParts = identifier.getVersion().split("\\.");
            int major = tryParse(versionParts, 0);
            int minor = tryParse(versionParts, 1);
            int patch = tryParse(versionParts, 2);
            return major==2 && (minor < 17 || (minor==17 && patch < 1));
        }

        private static int tryParse(String[] versionParts, int index) {
            try {
                return Integer.parseInt(versionParts[index]);
            } catch (Exception e) {
                return 0;
            }
        }
    }
}
