package com.gradle.publish;

import com.gradle.publish.protocols.v1.models.publish.ArtifactTypeCodec;
import com.gradle.publish.protocols.v1.models.publish.BuildMetadata;
import com.gradle.publish.protocols.v1.models.publish.NewVersionRequest;
import com.gradle.publish.protocols.v1.models.publish.PublishArtifact;
import com.gradle.publish.protocols.v1.models.publish.PublishMavenCoordinates;
import com.gradle.publish.protocols.v1.models.publish.PublishNewVersionRequest;
import com.gradle.publish.protocols.v1.models.publish.ValidateNewVersionRequest;
import org.gradle.api.DefaultTask;
import org.gradle.api.Project;
import org.gradle.api.Transformer;
import org.gradle.api.file.RegularFile;
import org.gradle.api.file.RegularFileProperty;
import org.gradle.api.logging.Logger;
import org.gradle.api.logging.Logging;
import org.gradle.api.provider.Property;
import org.gradle.api.provider.Provider;
import org.gradle.api.publish.PublishingExtension;
import org.gradle.api.publish.maven.MavenArtifact;
import org.gradle.api.publish.maven.internal.publication.DefaultMavenPublication;
import org.gradle.api.publish.maven.internal.publisher.MavenNormalizedPublication;
import org.gradle.api.publish.maven.tasks.GenerateMavenPom;
import org.gradle.api.publish.tasks.GenerateModuleMetadata;
import org.gradle.api.resources.MissingResourceException;
import org.gradle.api.tasks.Input;
import org.gradle.api.tasks.InputFiles;
import org.gradle.api.tasks.PathSensitive;
import org.gradle.api.tasks.PathSensitivity;
import org.gradle.api.tasks.TaskAction;
import org.gradle.api.tasks.options.Option;
import org.gradle.work.DisableCachingByDefault;
import org.jetbrains.annotations.NotNull;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.net.URI;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Properties;
import java.util.Set;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

import static java.lang.String.format;

@DisableCachingByDefault(because = "Not cacheable")
public abstract class PublishTask extends DefaultTask {
    private static final String SKIP_NAMESPACE_CHECK_PROPERTY = "gradle.publish.skip.namespace.check";
    private static final String MAVEN_PUBLISH_POM_TASK_NAME = "generatePomFileForPluginMavenPublication";
    private static final String MAVEN_PUBLISH_GMM_TASK_NAME = "generateMetadataFileForPluginMavenPublication";
    static final String MAVEN_PUBLICATION_NAME = "pluginMaven";

    private static final Logger LOGGER = Logging.getLogger(PublishTask.class);

    private final Provider<Credentials> credentials = CredentialsValueSource.create(getProject());

    @Input
    @Option(option = "validate-only", description = "Do not publish plugins, only validate they can be published")
    @org.gradle.api.tasks.Optional
    public abstract Property<Boolean> getValidateOnly();

    @Input
    public abstract Property<MavenNormalizedPublication> getNormalizedMavenPublication();

    @Input
    public abstract Property<PluginsConfigurations> getConfig();

    @InputFiles
    @org.gradle.api.tasks.Optional
    @PathSensitive(PathSensitivity.NONE)
    public abstract RegularFileProperty getPublishingLocation();

    @Input
    public abstract Property<String> getPluginPortalUrl();

    private final PortalPublisher portalPublisher = new PortalPublisher(
        getProject().getLayout().getProjectDirectory().getAsFile(),
        credentials,
        getPluginPortalUrl()
    );

    private final String gradleVersion = getProject().getGradle().getGradleVersion();
    private final Provider<Boolean> getSkipNamespaceCheck = getProject().getProviders()
        .systemProperty(SKIP_NAMESPACE_CHECK_PROPERTY)
        .map(new BooleanTransformer())
        .orElse(false);

    @TaskAction
    void executeTask() throws Exception {
        ConfigValidator validator = new ConfigValidator(getSkipNamespaceCheck.get());
        validator.validateConfig(getConfig().get());

        ensurePomIsAvailable();
        PublishMavenCoordinates mavenCoords = getPublishMavenCoordinates();
        validator.validateMavenCoordinates(mavenCoords.getGroupId(), mavenCoords.getArtifactId(), mavenCoords.getVersion());

        List<? extends NewVersionRequest> requests = buildPublishRequests(mavenCoords);
        validatePluginDescriptors(requests);

        Map<PublishArtifact, File> artifacts = collectArtifacts();

        if (getValidateOnly().getOrElse(false)) {
            portalPublisher.validatePublishingToPortal(cast(requests), mavenCoords, artifacts);
        } else {
            portalPublisher.publishToPortal(cast(requests), mavenCoords, artifacts);
        }
    }

    @SuppressWarnings("unchecked")
    private <T extends NewVersionRequest> List<T> cast(List<? extends NewVersionRequest> requests) {
        return (List<T>) requests;
    }

    private void validatePluginDescriptors(List<? extends NewVersionRequest> requests) {
        File artifactFile = findMainArtifact();
        try (ZipFile zip = new ZipFile(artifactFile)) {
            for (NewVersionRequest request : requests) {
                //TODO: we need to check artifacts of other variants too
                // but have the ability to specify that some artifact doesn't need it
                validatePluginDescriptor(zip, request.getPluginId());
            }
        } catch (IOException e) {
            throw new RuntimeException("Unable to validate plugin jar " + artifactFile.getPath(), e);
        }
    }

    private void validatePluginDescriptor(ZipFile zip, String pluginId) throws IOException {
        String resPath = format("META-INF/gradle-plugins/%s.properties", pluginId);
        ZipEntry descriptorEntry = zip.getEntry(resPath);
        if (descriptorEntry == null) {
            throw new IllegalArgumentException(
                format("No plugin descriptor for plugin ID '%s'.\nCreate a " +
                        "'META-INF/gradle-plugins/%s.properties' file with a " +
                        "'implementation-class' property pointing to the plugin class " +
                        "implementation.",
                    pluginId,
                    pluginId));
        }
        Properties descriptor = new Properties();
        descriptor.load(zip.getInputStream(descriptorEntry));
        String pluginClassName = descriptor.getProperty("implementation-class");
        if (pluginClassName == null || pluginClassName.trim().isEmpty()) {
            throw new IllegalArgumentException(
                format("Plugin descriptor for plugin ID '%s' does not specify a plugin\n"
                    + "class with the implementation-class property", pluginId));
        }

        String pluginClassResourcePath = pluginClassName.replace('.', '/').concat(".class");
        if (zip.getEntry(pluginClassResourcePath) == null) {
            throw new IllegalArgumentException(
                format("Plugin descriptor for plugin ID '%s' specifies a plugin\n"
                    + "class '%s' that is not present in the jar file", pluginId, pluginClassName));
        }
    }

    private void ensurePomIsAvailable() {
        RegularFile pomFile = getPublishingLocation().getOrNull();
        if (pomFile == null || !pomFile.getAsFile().exists()) {
            URI pomUri = (pomFile == null) ? null:pomFile.getAsFile().toURI();
            throw new MissingResourceException(pomUri, "Could not use POM from " + MAVEN_PUBLISH_POM_TASK_NAME + " task because it does not exist");
        }
    }

    private @NotNull PublishMavenCoordinates getPublishMavenCoordinates() {
        MavenNormalizedPublication pluginPublication = getNormalizedMavenPublication().get();
        String groupId = pluginPublication.getGroupId();
        String artifactId = pluginPublication.getArtifactId();
        String version = pluginPublication.getVersion();
        return new PublishMavenCoordinates(groupId, artifactId, version);
    }

    void addAndHashArtifact(Map<PublishArtifact, File> artifacts, File file, String type, String classifier) throws IOException {
        if (file != null && file.exists()) {
            try (FileInputStream fis = new FileInputStream(file)) {
                String hash = Hasher.hash(fis);
                try {
                    String artifactType = ArtifactTypeCodec.encode(type, classifier);
                    artifacts.put(new PublishArtifact(artifactType, hash), file);
                } catch (IllegalArgumentException e) {
                    LOGGER.warn("Ignoring unknown artifact with type \"{}\" and " +
                            "classifier \"{}\".\nYou can only upload normal jars, " +
                            "sources jars, javadoc jars and groovydoc jars\n" +
                            "with or without signatures to the Plugin Portal at this time.",
                        type, classifier);
                }
            }
        }
    }

    private Map<PublishArtifact, File> collectArtifacts() throws IOException {
        Map<PublishArtifact, File> artifacts = new LinkedHashMap<>();
        for (MavenArtifact mavenArtifact : getNormalizedMavenPublication().get().getAllArtifacts()) {
            addAndHashArtifact(artifacts, mavenArtifact.getFile(), mavenArtifact.getExtension(), mavenArtifact.getClassifier());
        }
        return artifacts;
    }

    private File findMainArtifact() {
        MavenNormalizedPublication publication = getNormalizedMavenPublication().get();

        if (isMainArtifact(publication.getMainArtifact())) {
            return publication.getMainArtifact().getFile();
        }

        Set<MavenArtifact> artifacts = publication.getAllArtifacts();
        Optional<MavenArtifact> artifact = artifacts.stream().filter(this::isMainArtifact).findFirst();
        if (artifact.isPresent()) {
            return artifact.get().getFile();
        }

        throw new IllegalArgumentException("Cannot determine main artifact to upload - could not find jar artifact with empty classifier");
    }

    private boolean isMainArtifact(MavenArtifact artifact) {
        if (artifact == null) {
            return false;
        }

        if (!"jar".equals(artifact.getExtension())) {
            return false;
        }

        String classifier = artifact.getClassifier();
        return classifier == null || classifier.trim().isEmpty();
    }

    private List<NewVersionRequest> buildPublishRequests(PublishMavenCoordinates mavenCoords) {
        PluginsConfigurations pluginConfig = getConfig().get();

        List<NewVersionRequest> reqs = new ArrayList<>();
        for (PluginConfiguration plugin : pluginConfig.plugins) {
            reqs.add(buildPublishRequest(mavenCoords, pluginConfig.website, pluginConfig.vcsUrl, plugin));
        }
        return reqs;
    }

    private NewVersionRequest buildPublishRequest(PublishMavenCoordinates mavenCoords, String website, String vcsUrl, PluginConfiguration plugin) {
        NewVersionRequest request;
        if (getValidateOnly().getOrElse(false)) {
            request = new ValidateNewVersionRequest();
        } else {
            request = new PublishNewVersionRequest();
        }
        BuildMetadata buildMetadata = new BuildMetadata(gradleVersion);
        request.setBuildMetadata(buildMetadata);

        request.setPluginId(plugin.id);

        request.setPluginVersion(mavenCoords.getVersion());
        request.setDisplayName(plugin.displayName);

        request.setDescription(plugin.description);
        request.setTags(plugin.tags);
        request.setWebSite(website);
        request.setVcsUrl(vcsUrl);

        return request;
    }

    public void dependOnPublishTasks() {
        Project project = getProject();

        GenerateMavenPom pomTask = (GenerateMavenPom) getProject().getTasks().findByName(MAVEN_PUBLISH_POM_TASK_NAME);
        if (pomTask != null) {
            this.getPublishingLocation().convention(pomTask::getDestination);
        }
        dependsOn(pomTask);

        GenerateModuleMetadata gmmTask = (GenerateModuleMetadata) project.getTasks().findByName(MAVEN_PUBLISH_GMM_TASK_NAME);
        if (gmmTask != null && gmmTask.getEnabled()) {
            dependsOn(gmmTask.getOutputFile());
        }

        PublishingExtension publishing = getProject().getExtensions().getByType(PublishingExtension.class);
        getNormalizedMavenPublication().set(project.provider(() ->
            ((DefaultMavenPublication) publishing.getPublications().getByName(PublishTask.MAVEN_PUBLICATION_NAME)).asNormalisedPublication()
        ));
    }

    // workaround for Gradle 7.4 issue when transformer lamba is not serializable
    private static final class BooleanTransformer implements Transformer<Boolean, String> {
        @Override
        public Boolean transform(@NotNull String original) {
            return Boolean.parseBoolean(original);
        }
    }
}
