/*
 * Decompiled with CFR 0.152.
 */
package dev.sigstore;

import com.google.api.client.util.Preconditions;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.hash.Hashing;
import com.google.common.io.Files;
import com.google.gson.Gson;
import com.google.protobuf.InvalidProtocolBufferException;
import com.google.protobuf.Message;
import dev.sigstore.KeylessVerificationException;
import dev.sigstore.TrustedRootProvider;
import dev.sigstore.VerificationOptions;
import dev.sigstore.bundle.Bundle;
import dev.sigstore.dsse.InTotoPayload;
import dev.sigstore.encryption.certificates.Certificates;
import dev.sigstore.encryption.signers.Verifiers;
import dev.sigstore.fulcio.client.FulcioVerificationException;
import dev.sigstore.fulcio.client.FulcioVerifier;
import dev.sigstore.json.ProtoJson;
import dev.sigstore.proto.common.v1.HashAlgorithm;
import dev.sigstore.proto.rekor.v2.DSSELogEntryV002;
import dev.sigstore.proto.rekor.v2.HashedRekordLogEntryV002;
import dev.sigstore.proto.rekor.v2.Signature;
import dev.sigstore.proto.rekor.v2.Verifier;
import dev.sigstore.rekor.client.HashedRekordRequest;
import dev.sigstore.rekor.client.RekorEntry;
import dev.sigstore.rekor.client.RekorTypeException;
import dev.sigstore.rekor.client.RekorTypes;
import dev.sigstore.rekor.client.RekorVerificationException;
import dev.sigstore.rekor.client.RekorVerifier;
import dev.sigstore.rekor.dsse.v0_0_1.Dsse;
import dev.sigstore.rekor.dsse.v0_0_1.PayloadHash;
import dev.sigstore.timestamp.client.ImmutableTimestampResponse;
import dev.sigstore.timestamp.client.TimestampException;
import dev.sigstore.timestamp.client.TimestampVerificationException;
import dev.sigstore.timestamp.client.TimestampVerifier;
import dev.sigstore.trustroot.SigstoreConfigurationException;
import dev.sigstore.trustroot.SigstoreTrustedRoot;
import dev.sigstore.tuf.SigstoreTufClient;
import java.io.File;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Path;
import java.security.InvalidAlgorithmParameterException;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.security.SignatureException;
import java.security.cert.CertPath;
import java.security.cert.CertificateEncodingException;
import java.security.cert.CertificateException;
import java.security.cert.CertificateExpiredException;
import java.security.cert.CertificateNotYetValidException;
import java.security.cert.X509Certificate;
import java.security.spec.InvalidKeySpecException;
import java.util.Arrays;
import java.util.Base64;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;
import org.bouncycastle.util.encoders.DecoderException;
import org.bouncycastle.util.encoders.Hex;

public class KeylessVerifier {
    private final FulcioVerifier fulcioVerifier;
    private final RekorVerifier rekorVerifier;
    private final TimestampVerifier timestampVerifier;

    private KeylessVerifier(FulcioVerifier fulcioVerifier, RekorVerifier rekorVerifier, TimestampVerifier timestampVerifier) {
        this.fulcioVerifier = fulcioVerifier;
        this.rekorVerifier = rekorVerifier;
        this.timestampVerifier = timestampVerifier;
    }

    public static Builder builder() {
        return new Builder();
    }

    public void verify(Path artifact, Bundle bundle, VerificationOptions options) throws KeylessVerificationException {
        try {
            byte[] artifactDigest = Files.asByteSource((File)artifact.toFile()).hash(Hashing.sha256()).asBytes();
            this.verify(artifactDigest, bundle, options);
        }
        catch (IOException e) {
            throw new KeylessVerificationException("Could not hash provided artifact path: " + String.valueOf(artifact));
        }
    }

    public void verify(byte[] artifactDigest, Bundle bundle, VerificationOptions options) throws KeylessVerificationException {
        byte[] signature;
        if (bundle.getDsseEnvelope().isEmpty() && bundle.getMessageSignature().isEmpty()) {
            throw new IllegalStateException("Bundle must contain a message signature or DSSE envelope to verify");
        }
        CertPath signingCert = bundle.getCertPath();
        X509Certificate leafCert = Certificates.getLeaf(signingCert);
        try {
            this.fulcioVerifier.verifySigningCertificate(signingCert);
        }
        catch (FulcioVerificationException | IOException ex) {
            throw new KeylessVerificationException("Fulcio certificate was not valid: " + ex.getMessage(), ex);
        }
        this.checkCertificateMatchers(leafCert, options.getCertificateMatchers());
        RekorEntry rekorEntry = bundle.getEntries().get(0);
        if (bundle.getMessageSignature().isPresent()) {
            Bundle.MessageSignature messageSignature = bundle.getMessageSignature().get();
            this.checkMessageSignature(messageSignature, rekorEntry, artifactDigest, leafCert);
            signature = messageSignature.getSignature();
        } else {
            Bundle.DsseEnvelope dsseEnvelope = bundle.getDsseEnvelope().get();
            this.checkDsseEnvelope(rekorEntry, dsseEnvelope, artifactDigest, leafCert);
            signature = dsseEnvelope.getSignature();
        }
        this.verifyTimestamps(leafCert, bundle.getTimestamps(), signature);
        try {
            this.rekorVerifier.verifyEntry(rekorEntry);
        }
        catch (RekorVerificationException ex) {
            throw new KeylessVerificationException("Transparency log entry could not be verified", ex);
        }
    }

    private void verifyTimestamps(X509Certificate leafCert, List<Bundle.Timestamp> timestamps, byte[] signature) throws KeylessVerificationException {
        if (timestamps == null || timestamps.isEmpty()) {
            return;
        }
        for (Bundle.Timestamp timestamp : timestamps) {
            byte[] tsBytes = timestamp.getRfc3161Timestamp();
            if (tsBytes == null || tsBytes.length == 0) {
                throw new KeylessVerificationException("Found an empty or null RFC3161 timestamp in bundle");
            }
            try {
                ImmutableTimestampResponse tsResp = ImmutableTimestampResponse.builder().encoded(tsBytes).build();
                this.timestampVerifier.verify(tsResp, signature);
                leafCert.checkValidity(tsResp.getGenTime());
            }
            catch (TimestampException | TimestampVerificationException | CertificateExpiredException | CertificateNotYetValidException e) {
                throw new KeylessVerificationException("RFC3161 timestamp verification failed: " + e.getMessage(), e);
            }
        }
    }

    @VisibleForTesting
    void checkCertificateMatchers(X509Certificate cert, List<VerificationOptions.CertificateMatcher> matchers) throws KeylessVerificationException {
        try {
            if (matchers.size() > 0 && matchers.stream().noneMatch(matcher -> matcher.test(cert))) {
                String matcherSpec = matchers.stream().map(Object::toString).collect(Collectors.joining(",", "[", "]"));
                throw new KeylessVerificationException("No provided certificate identities matched values in certificate: " + matcherSpec);
            }
        }
        catch (VerificationOptions.UncheckedCertificateException ce) {
            throw new KeylessVerificationException("Could not verify certificate identities: " + ce.getMessage());
        }
    }

    private void checkMessageSignature(Bundle.MessageSignature messageSignature, RekorEntry rekorEntry, byte[] artifactDigest, X509Certificate leafCert) throws KeylessVerificationException {
        block20: {
            byte[] bundleDigest;
            if (messageSignature.getMessageDigest().isPresent() && !Arrays.equals(artifactDigest, bundleDigest = messageSignature.getMessageDigest().get().getDigest())) {
                throw new KeylessVerificationException("Provided artifact digest does not match digest used for verification\nprovided(hex) : " + Hex.toHexString((byte[])artifactDigest) + "\nverification(hex) : " + Hex.toHexString((byte[])bundleDigest));
            }
            byte[] signature = messageSignature.getSignature();
            try {
                if (!Verifiers.newVerifier(leafCert.getPublicKey()).verifyDigest(artifactDigest, signature)) {
                    throw new KeylessVerificationException("Artifact signature was not valid");
                }
            }
            catch (InvalidKeyException | NoSuchAlgorithmException ex) {
                throw new RuntimeException(ex);
            }
            catch (SignatureException ex) {
                throw new KeylessVerificationException("Signature could not be processed: " + ex.getMessage(), ex);
            }
            String version = rekorEntry.getBodyDecoded().getApiVersion();
            if ("0.0.1".equals(version)) {
                try {
                    RekorTypes.getHashedRekord(rekorEntry);
                    String calculatedHashedRekord = HashedRekordRequest.newHashedRekordRequest(artifactDigest, Certificates.toPemBytes(leafCert), signature).toJsonPayload();
                    String body = new String(Base64.getDecoder().decode(rekorEntry.getBody()), StandardCharsets.UTF_8);
                    if (!Objects.equals(calculatedHashedRekord, body)) {
                        throw new KeylessVerificationException("Provided verification materials are inconsistent with log entry");
                    }
                    break block20;
                }
                catch (IOException e) {
                    throw new RuntimeException("Unexpected IOException on valid leafCert", e);
                }
                catch (RekorTypeException re) {
                    throw new KeylessVerificationException("Unexpected rekor type", re);
                }
            }
            if ("0.0.2".equals(version)) {
                HashedRekordLogEntryV002 logEntrySpec;
                try {
                    HashedRekordLogEntryV002.Builder builder = HashedRekordLogEntryV002.newBuilder();
                    ProtoJson.parser().ignoringUnknownFields().merge(new Gson().toJson(rekorEntry.getBodyDecoded().getSpec().getAsJsonObject().get("hashedRekordV002")), (Message.Builder)builder);
                    logEntrySpec = builder.build();
                }
                catch (InvalidProtocolBufferException ipbe) {
                    throw new KeylessVerificationException("Could not parse HashedRekordLogEntryV002 from log entry body");
                }
                if (!logEntrySpec.getData().getAlgorithm().equals((Object)HashAlgorithm.SHA2_256)) {
                    throw new KeylessVerificationException("Unsupported digest algorithm in log entry: " + String.valueOf((Object)logEntrySpec.getData().getAlgorithm()));
                }
                if (!Arrays.equals(logEntrySpec.getData().getDigest().toByteArray(), artifactDigest)) {
                    throw new KeylessVerificationException("Artifact digest does not match digest in log entry spec");
                }
                if (!Arrays.equals(logEntrySpec.getSignature().getContent().toByteArray(), signature)) {
                    throw new KeylessVerificationException("Signature does not match signature in log entry spec");
                }
                Verifier verifier = logEntrySpec.getSignature().getVerifier();
                if (!verifier.hasX509Certificate()) {
                    throw new KeylessVerificationException("Rekor entry verifier is missing X.509 certificate");
                }
                try {
                    byte[] certFromRekor = verifier.getX509Certificate().getRawBytes().toByteArray();
                    byte[] certFromBundle = leafCert.getEncoded();
                    if (!Arrays.equals(certFromRekor, certFromBundle)) {
                        throw new KeylessVerificationException("Certificate in rekor entry does not match certificate in bundle");
                    }
                    break block20;
                }
                catch (CertificateEncodingException e) {
                    throw new KeylessVerificationException("Could not encode leaf certificate for comparison", e);
                }
            }
            throw new KeylessVerificationException("Unsupported hashedrekord version");
        }
    }

    private void checkDsseEnvelope(RekorEntry rekorEntry, Bundle.DsseEnvelope dsseEnvelope, byte[] artifactDigest, X509Certificate leafCert) throws KeylessVerificationException {
        if (!Objects.equals("application/vnd.in-toto+json", dsseEnvelope.getPayloadType())) {
            throw new KeylessVerificationException("DSSE envelope must have payload type application/vnd.in-toto+json, but found '" + dsseEnvelope.getPayloadType() + "'");
        }
        InTotoPayload payload = InTotoPayload.from(dsseEnvelope);
        if (payload.getSubject().stream().noneMatch(subject -> {
            if (subject.getDigest().containsKey("sha256")) {
                try {
                    byte[] digestBytes = Hex.decode((String)subject.getDigest().get("sha256"));
                    return Arrays.equals(artifactDigest, digestBytes);
                }
                catch (DecoderException decoderException) {
                    // empty catch block
                }
            }
            return false;
        })) {
            String providedHashes = payload.getSubject().stream().map(s -> s.getDigest().getOrDefault("sha256", "no-sha256-hash")).collect(Collectors.joining(",", "[", "]"));
            throw new KeylessVerificationException("Provided artifact digest does not match any subject sha256 digests in DSSE payload\nprovided(hex) : " + Hex.toHexString((byte[])artifactDigest) + "\nverification  : " + providedHashes);
        }
        if (dsseEnvelope.getSignatures().size() != 1) {
            throw new KeylessVerificationException("DSSE envelope must have exactly 1 signature, but found: " + dsseEnvelope.getSignatures().size());
        }
        try {
            if (!Verifiers.newVerifier(leafCert.getPublicKey()).verify(dsseEnvelope.getPAE(), dsseEnvelope.getSignature())) {
                throw new KeylessVerificationException("DSSE signature was not valid");
            }
        }
        catch (InvalidKeyException | NoSuchAlgorithmException ex) {
            throw new RuntimeException(ex);
        }
        catch (SignatureException se) {
            throw new KeylessVerificationException("Signature could not be processed", se);
        }
        String version = rekorEntry.getBodyDecoded().getApiVersion();
        if ("0.0.1".equals(version)) {
            byte[] payloadDigest;
            Dsse rekorDsse;
            try {
                rekorDsse = RekorTypes.getDsse(rekorEntry);
            }
            catch (RekorTypeException re) {
                throw new KeylessVerificationException("Unexpected rekor type", re);
            }
            PayloadHash.Algorithm algorithm = rekorDsse.getPayloadHash().getAlgorithm();
            if (algorithm != PayloadHash.Algorithm.SHA_256) {
                throw new KeylessVerificationException("Cannot process DSSE entry with hashing algorithm " + algorithm.toString());
            }
            try {
                payloadDigest = Hex.decode((String)rekorDsse.getPayloadHash().getValue());
            }
            catch (DecoderException de) {
                throw new KeylessVerificationException("Could not decode hex sha256 artifact hash in hashrekord", de);
            }
            byte[] calculatedDigest = Hashing.sha256().hashBytes(dsseEnvelope.getPayload()).asBytes();
            if (!Arrays.equals(calculatedDigest, payloadDigest)) {
                throw new KeylessVerificationException("Digest of DSSE payload in bundle does not match DSSE payload digest in log entry");
            }
            if (rekorDsse.getSignatures().size() != 1) {
                throw new KeylessVerificationException("DSSE log entry must have exactly 1 signature, but found: " + rekorDsse.getSignatures().size());
            }
            if (!Base64.getEncoder().encodeToString(dsseEnvelope.getSignature()).equals(rekorDsse.getSignatures().get(0).getSignature())) {
                throw new KeylessVerificationException("Provided DSSE signature materials are inconsistent with DSSE log entry");
            }
        } else if ("0.0.2".equals(version)) {
            DSSELogEntryV002 logEntrySpec;
            try {
                DSSELogEntryV002.Builder builder = DSSELogEntryV002.newBuilder();
                ProtoJson.parser().merge(new Gson().toJson(rekorEntry.getBodyDecoded().getSpec().getAsJsonObject().get("dsseV002")), (Message.Builder)builder);
                logEntrySpec = builder.build();
            }
            catch (InvalidProtocolBufferException ipbe) {
                throw new KeylessVerificationException("Could not parse DSSELogEntryV002 from log entry body", ipbe);
            }
            if (!logEntrySpec.getPayloadHash().getAlgorithm().equals((Object)HashAlgorithm.SHA2_256)) {
                throw new KeylessVerificationException("Unsupported digest algorithm in log entry: " + String.valueOf((Object)logEntrySpec.getPayloadHash().getAlgorithm()));
            }
            byte[] calculatedDigest = Hashing.sha256().hashBytes(dsseEnvelope.getPayload()).asBytes();
            if (!Arrays.equals(logEntrySpec.getPayloadHash().getDigest().toByteArray(), calculatedDigest)) {
                throw new KeylessVerificationException("Digest of DSSE payload in bundle does not match DSSE payload digest in log entry");
            }
            if (logEntrySpec.getSignaturesCount() != 1) {
                throw new KeylessVerificationException("Log entry spec must have exactly 1 signature, but found: " + logEntrySpec.getSignaturesCount());
            }
            Signature logSignature = logEntrySpec.getSignatures(0);
            if (!Arrays.equals(dsseEnvelope.getSignature(), logSignature.getContent().toByteArray())) {
                throw new KeylessVerificationException("Signature in DSSE envelope does not match signature in log entry spec");
            }
            Verifier verifier = logSignature.getVerifier();
            if (!verifier.hasX509Certificate()) {
                throw new KeylessVerificationException("Rekor entry DSSE verifier is missing X.509 certificate");
            }
            try {
                byte[] certFromRekor = verifier.getX509Certificate().getRawBytes().toByteArray();
                byte[] certFromBundle = leafCert.getEncoded();
                if (!Arrays.equals(certFromRekor, certFromBundle)) {
                    throw new KeylessVerificationException("Certificate in rekor entry does not match certificate in bundle");
                }
            }
            catch (CertificateEncodingException e) {
                throw new KeylessVerificationException("Could not encode leaf certificate for comparison", e);
            }
        }
    }

    public static class Builder {
        private TrustedRootProvider trustedRootProvider;

        public KeylessVerifier build() throws InvalidAlgorithmParameterException, CertificateException, InvalidKeySpecException, NoSuchAlgorithmException, SigstoreConfigurationException {
            Preconditions.checkNotNull((Object)this.trustedRootProvider);
            SigstoreTrustedRoot trustedRoot = this.trustedRootProvider.get();
            FulcioVerifier fulcioVerifier = FulcioVerifier.newFulcioVerifier(trustedRoot);
            RekorVerifier rekorVerifier = RekorVerifier.newRekorVerifier(trustedRoot);
            TimestampVerifier timestampVerifier = TimestampVerifier.newTimestampVerifier(trustedRoot);
            return new KeylessVerifier(fulcioVerifier, rekorVerifier, timestampVerifier);
        }

        public Builder sigstorePublicDefaults() {
            SigstoreTufClient.Builder sigstoreTufClientBuilder = SigstoreTufClient.builder().usePublicGoodInstance();
            this.trustedRootProvider = TrustedRootProvider.from(sigstoreTufClientBuilder);
            return this;
        }

        public Builder sigstoreStagingDefaults() {
            SigstoreTufClient.Builder sigstoreTufClientBuilder = SigstoreTufClient.builder().useStagingInstance();
            this.trustedRootProvider = TrustedRootProvider.from(sigstoreTufClientBuilder);
            return this;
        }

        public Builder trustedRootProvider(TrustedRootProvider trustedRootProvider) {
            this.trustedRootProvider = trustedRootProvider;
            return this;
        }
    }
}

