/*
 * Decompiled with CFR 0.152.
 */
package dev.sigstore.timestamp.client;

import com.google.common.hash.Hashing;
import dev.sigstore.encryption.certificates.Certificates;
import dev.sigstore.timestamp.client.HashAlgorithm;
import dev.sigstore.timestamp.client.TimestampException;
import dev.sigstore.timestamp.client.TimestampResponse;
import dev.sigstore.timestamp.client.TimestampVerificationException;
import dev.sigstore.timestamp.client.UnsupportedHashAlgorithmException;
import dev.sigstore.trustroot.CertificateAuthority;
import dev.sigstore.trustroot.SigstoreTrustedRoot;
import java.io.IOException;
import java.security.InvalidAlgorithmParameterException;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.Provider;
import java.security.Security;
import java.security.cert.CertPath;
import java.security.cert.CertPathValidator;
import java.security.cert.CertPathValidatorException;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.cert.PKIXParameters;
import java.security.cert.X509Certificate;
import java.security.spec.InvalidKeySpecException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.stream.Collectors;
import org.bouncycastle.asn1.ASN1ObjectIdentifier;
import org.bouncycastle.cert.X509CertificateHolder;
import org.bouncycastle.cert.jcajce.JcaX509CertificateConverter;
import org.bouncycastle.cms.SignerInformationVerifier;
import org.bouncycastle.cms.jcajce.JcaSimpleSignerInfoVerifierBuilder;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.operator.OperatorCreationException;
import org.bouncycastle.tsp.TSPException;
import org.bouncycastle.tsp.TimeStampResponse;
import org.bouncycastle.tsp.TimeStampToken;
import org.bouncycastle.util.Store;

public class TimestampVerifier {
    private final List<CertificateAuthority> tsas;

    public static TimestampVerifier newTimestampVerifier(SigstoreTrustedRoot trustedRoot) throws InvalidAlgorithmParameterException, CertificateException, InvalidKeySpecException, NoSuchAlgorithmException {
        return TimestampVerifier.newTimestampVerifier(trustedRoot.getTSAs());
    }

    public static TimestampVerifier newTimestampVerifier(List<CertificateAuthority> tsas) throws InvalidKeySpecException, NoSuchAlgorithmException, InvalidAlgorithmParameterException, CertificateException {
        for (CertificateAuthority tsa : tsas) {
            tsa.asTrustAnchor();
        }
        return new TimestampVerifier(tsas);
    }

    private TimestampVerifier(List<CertificateAuthority> tsas) {
        this.tsas = tsas;
    }

    public void verify(TimestampResponse tsResp, byte[] artifact) throws TimestampVerificationException {
        byte[] artifactDigest;
        HashAlgorithm hashAlgorithm;
        TimeStampResponse bcTsResp;
        try {
            bcTsResp = new TimeStampResponse(tsResp.getEncoded());
        }
        catch (IOException | TSPException e) {
            throw new TimestampVerificationException("Failed to parse TimeStampResponse", e);
        }
        TimeStampToken tsToken = bcTsResp.getTimeStampToken();
        if (tsToken == null) {
            throw new TimestampVerificationException("No TimeStampToken found in response");
        }
        LinkedHashMap<String, String> tsaVerificationFailure = new LinkedHashMap<String, String>();
        Store tsCertStore = tsToken.getCertificates();
        boolean hasEmbeddedCerts = false;
        if (tsCertStore != null) {
            hasEmbeddedCerts = !tsCertStore.getMatches(null).isEmpty();
        }
        CertificateAuthority tsa = hasEmbeddedCerts ? this.findVerifyingTsaFromEmbeddedCerts(tsToken) : this.findVerifyingTsaByLeafSignature(tsToken);
        try {
            this.validateTsaChain(tsa, tsToken.getTimeStampInfo().getGenTime());
        }
        catch (TimestampException | InvalidAlgorithmParameterException | NoSuchProviderException | CertPathValidatorException e) {
            throw new TimestampVerificationException("Failed to validate TSA chain", e);
        }
        if (!tsa.getValidFor().contains(tsToken.getTimeStampInfo().getGenTime().toInstant())) {
            tsaVerificationFailure.put(tsa.getUri().toString(), "Timestamp generation time is not within TSA's validity period.");
            String errors = tsaVerificationFailure.entrySet().stream().map(entry -> (String)entry.getKey() + " (" + (String)entry.getValue() + ")").collect(Collectors.joining("\n"));
            throw new TimestampVerificationException("Certificate was not verifiable against TSAs\n" + errors);
        }
        ASN1ObjectIdentifier oid = tsToken.getTimeStampInfo().getMessageImprintAlgOID();
        try {
            hashAlgorithm = HashAlgorithm.from(oid);
        }
        catch (UnsupportedHashAlgorithmException e) {
            throw new TimestampVerificationException(e);
        }
        switch (hashAlgorithm) {
            case SHA256: {
                artifactDigest = Hashing.sha256().hashBytes(artifact).asBytes();
                break;
            }
            case SHA384: {
                artifactDigest = Hashing.sha384().hashBytes(artifact).asBytes();
                break;
            }
            case SHA512: {
                artifactDigest = Hashing.sha512().hashBytes(artifact).asBytes();
                break;
            }
            default: {
                throw new IllegalStateException();
            }
        }
        this.validateTokenMessageImprintDigest(tsToken, artifactDigest);
    }

    private void validateTokenSignature(TimeStampToken token, X509Certificate signingCert) throws TimestampVerificationException {
        try {
            JcaSimpleSignerInfoVerifierBuilder verifierBuilder = new JcaSimpleSignerInfoVerifierBuilder();
            SignerInformationVerifier verifier = verifierBuilder.setProvider("BC").build(signingCert);
            token.validate(verifier);
        }
        catch (OperatorCreationException oce) {
            throw new TimestampVerificationException("Failed to build SignerInformationVerifier", oce);
        }
        catch (TSPException tspe) {
            throw new TimestampVerificationException("Failed to validate TimeStampToken", tspe);
        }
    }

    private void validateTokenMessageImprintDigest(TimeStampToken token, byte[] artifactDigest) throws TimestampVerificationException {
        byte[] messageImprintDigest = token.getTimeStampInfo().getMessageImprintDigest();
        if (!Arrays.equals(messageImprintDigest, artifactDigest)) {
            throw new TimestampVerificationException("Timestamp message imprint digest does not match artifact hash");
        }
    }

    void validateTsaChain(CertificateAuthority tsa, Date tsDate) throws TimestampException, NoSuchProviderException, CertPathValidatorException, InvalidAlgorithmParameterException {
        PKIXParameters pkixParams;
        CertPathValidator cpv;
        try {
            cpv = CertPathValidator.getInstance("PKIX");
        }
        catch (NoSuchAlgorithmException e) {
            throw new RuntimeException("No PKIX CertPathValidator, we probably shouldn't be here, but this seems to be a system library error not a program control flow issue", e);
        }
        try {
            pkixParams = new PKIXParameters(Collections.singleton(tsa.asTrustAnchor()));
        }
        catch (InvalidAlgorithmParameterException | CertificateException e) {
            throw new RuntimeException("Can't create PKIX parameters for the TSA. This should have been checked when generating a verifier instance", e);
        }
        pkixParams.setRevocationEnabled(false);
        pkixParams.setDate(tsDate);
        cpv.validate(tsa.getCertPath(), pkixParams);
    }

    CertificateAuthority findVerifyingTsaFromEmbeddedCerts(TimeStampToken tsToken) throws TimestampVerificationException {
        CertPath tsCertPath;
        Store tsCertStore = tsToken.getCertificates();
        ArrayList<Certificate> tsCerts = new ArrayList<Certificate>();
        Collection tsCertHolders = tsCertStore.getMatches(null);
        for (X509CertificateHolder tsCertHolder : tsCertHolders) {
            JcaX509CertificateConverter converter = new JcaX509CertificateConverter().setProvider("BC");
            try {
                X509Certificate cert = converter.getCertificate(tsCertHolder);
                tsCerts.add(cert);
            }
            catch (CertificateException ce) {
                throw new TimestampVerificationException("Unable to convert certificate to X509Certificate", ce);
            }
        }
        try {
            tsCertPath = Certificates.toCertPath(tsCerts);
        }
        catch (CertificateException ce) {
            throw new TimestampVerificationException("Cannot convert certificates to CertPath", ce);
        }
        LinkedHashMap<String, String> tsaVerificationFailure = new LinkedHashMap<String, String>();
        for (CertificateAuthority tsa : this.tsas) {
            CertPath tsaChain = tsa.getCertPath();
            if (Certificates.getLeaf(tsCertPath).equals(Certificates.getLeaf(tsaChain))) {
                this.validateTokenSignature(tsToken, Certificates.getLeaf(tsCertPath));
                return tsa;
            }
            tsaVerificationFailure.put(tsa.getUri().toString(), "Embedded leaf certificate does not match this trusted TSA's leaf.");
        }
        String errors = tsaVerificationFailure.entrySet().stream().map(entry -> (String)entry.getKey() + " (" + (String)entry.getValue() + ")").collect(Collectors.joining("\n"));
        throw new TimestampVerificationException("Certificates in token were not verifiable against TSAs\n" + errors);
    }

    CertificateAuthority findVerifyingTsaByLeafSignature(TimeStampToken tsToken) throws TimestampVerificationException {
        LinkedHashMap<String, String> tsaVerificationFailure = new LinkedHashMap<String, String>();
        for (CertificateAuthority tsa : this.tsas) {
            CertPath tsaChain = tsa.getCertPath();
            X509Certificate tsaLeaf = Certificates.getLeaf(tsaChain);
            try {
                this.validateTokenSignature(tsToken, tsaLeaf);
                return tsa;
            }
            catch (TimestampVerificationException tsve) {
                tsaVerificationFailure.put(tsa.getUri().toString(), tsve.getMessage());
            }
        }
        String errors = tsaVerificationFailure.entrySet().stream().map(entry -> (String)entry.getKey() + " (" + (String)entry.getValue() + ")").collect(Collectors.joining("\n"));
        throw new TimestampVerificationException("Certificates in token were not verifiable against TSAs\n" + errors);
    }

    static {
        Security.addProvider((Provider)new BouncyCastleProvider());
    }
}

