/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.impl.biome;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import net.fabricmc.fabric.api.biome.v1.OverworldClimate;
import net.fabricmc.fabric.impl.biome.VariantTransformer.SubTransformer;
import net.minecraft.class_1959;
import net.minecraft.class_3630;
import net.minecraft.class_5321;

/**
 * Deals with picking variants for you.
 */
final class VariantTransformer {
	private final SubTransformer defaultTransformer = new SubTransformer();
	private final Map<OverworldClimate, SubTransformer> transformers = new HashMap<>();

	/**
	 * @param variant the variant that the replaced biome is replaced with
	 * @param chance the chance of replacement of the biome into the variant
	 * @param climates the climates that the variant can replace the base biome in, empty/null indicates all climates
	 */
	void addBiome(class_5321<class_1959> variant, double chance, OverworldClimate[] climates) {
		if (climates == null || climates.length == 0) {
			defaultTransformer.addBiome(variant, chance);
			climates = OverworldClimate.values();
		}

		for (OverworldClimate climate : climates) {
			transformers.computeIfAbsent(climate, c -> new SubTransformer()).addBiome(variant, chance);
		}
	}

	/**
	 * Transforms a biome into a variant randomly depending on its chance.
	 *
	 * @param replaced biome to transform
	 * @param random the {@link class_3630} from the layer
	 * @return the transformed biome
	 */
	class_5321<class_1959> transformBiome(class_5321<class_1959> replaced, class_3630 random, OverworldClimate climate) {
		if (climate == null) {
			return defaultTransformer.transformBiome(replaced, random);
		}

		SubTransformer transformer = transformers.get(climate);

		if (transformer != null) {
			return transformer.transformBiome(replaced, random);
		} else {
			return replaced;
		}
	}

	static final class SubTransformer {
		private final List<BiomeVariant> variants = new ArrayList<>();

		/**
		 * @param variant the variant that the replaced biome is replaced with
		 * @param chance the chance of replacement of the biome into the variant
		 */
		private void addBiome(class_5321<class_1959> variant, double chance) {
			variants.add(new BiomeVariant(variant, chance));
		}

		/**
		 * Transforms a biome into a variant randomly depending on its chance.
		 *
		 * @param replaced biome to transform
		 * @param random the {@link class_3630} from the layer
		 * @return the transformed biome
		 */
		private class_5321<class_1959> transformBiome(class_5321<class_1959> replaced, class_3630 random) {
			for (BiomeVariant variant : variants) {
				if (random.method_15834(Integer.MAX_VALUE) < variant.getChance() * Integer.MAX_VALUE) {
					return variant.getVariant();
				}
			}

			return replaced;
		}
	}
}
