/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.impl.client.indigo.renderer.render;

import java.util.function.Supplier;

import org.jetbrains.annotations.Nullable;
import net.fabricmc.fabric.api.renderer.v1.material.BlendMode;
import net.fabricmc.fabric.api.renderer.v1.material.RenderMaterial;
import net.fabricmc.fabric.api.util.TriState;
import net.fabricmc.fabric.impl.client.indigo.renderer.helper.ColorHelper;
import net.fabricmc.fabric.impl.client.indigo.renderer.mesh.MutableQuadViewImpl;
import net.fabricmc.fabric.mixin.client.indigo.renderer.ItemRendererAccessor;
import net.minecraft.class_1087;
import net.minecraft.class_1799;
import net.minecraft.class_1921;
import net.minecraft.class_2350;
import net.minecraft.class_325;
import net.minecraft.class_4587;
import net.minecraft.class_4588;
import net.minecraft.class_4597;
import net.minecraft.class_4696;
import net.minecraft.class_4722;
import net.minecraft.class_5819;
import net.minecraft.class_765;
import net.minecraft.class_7837;
import net.minecraft.class_811;
import net.minecraft.class_918;

/**
 * The render context used for item rendering.
 */
public class ItemRenderContext extends AbstractRenderContext {
	/** Value vanilla uses for item rendering.  The only sensible choice, of course.  */
	private static final long ITEM_RANDOM_SEED = 42L;

	private final class_325 colorMap;
	private final class_5819 random = class_5819.method_43047();
	private final Supplier<class_5819> randomSupplier = () -> {
		random.method_43052(ITEM_RANDOM_SEED);
		return random;
	};

	private class_1799 itemStack;
	private class_811 transformMode;
	private class_4587 matrixStack;
	private class_4597 vertexConsumerProvider;
	private int lightmap;

	private boolean isDefaultTranslucent;
	private boolean isDefaultGlint;
	private boolean isGlintDynamicDisplay;

	private class_4587.class_4665 dynamicDisplayGlintEntry;
	private class_4588 translucentVertexConsumer;
	private class_4588 cutoutVertexConsumer;
	private class_4588 translucentGlintVertexConsumer;
	private class_4588 cutoutGlintVertexConsumer;

	public ItemRenderContext(class_325 colorMap) {
		this.colorMap = colorMap;
	}

	@Override
	public boolean isFaceCulled(@Nullable class_2350 face) {
		throw new IllegalStateException("isFaceCulled can only be called on a block render context.");
	}

	@Override
	public class_811 itemTransformationMode() {
		return transformMode;
	}

	public void renderModel(class_1799 itemStack, class_811 transformMode, class_4587 matrixStack, class_4597 vertexConsumerProvider, int lightmap, int overlay, class_1087 model) {
		this.itemStack = itemStack;
		this.transformMode = transformMode;
		this.matrixStack = matrixStack;
		this.vertexConsumerProvider = vertexConsumerProvider;
		this.lightmap = lightmap;
		this.overlay = overlay;
		computeOutputInfo();

		matrix = matrixStack.method_23760().method_23761();
		normalMatrix = matrixStack.method_23760().method_23762();

		model.emitItemQuads(itemStack, randomSupplier, this);

		this.itemStack = null;
		this.matrixStack = null;
		this.vertexConsumerProvider = null;

		dynamicDisplayGlintEntry = null;
		translucentVertexConsumer = null;
		cutoutVertexConsumer = null;
		translucentGlintVertexConsumer = null;
		cutoutGlintVertexConsumer = null;
	}

	private void computeOutputInfo() {
		isDefaultTranslucent = class_4696.method_23678(itemStack) == class_4722.method_29382();
		isDefaultGlint = itemStack.method_7958();
		isGlintDynamicDisplay = ItemRendererAccessor.fabric_callUsesDynamicDisplay(itemStack);
	}

	@Override
	protected void renderQuad(MutableQuadViewImpl quad) {
		if (!transform(quad)) {
			return;
		}

		final RenderMaterial mat = quad.material();
		final int colorIndex = mat.disableColorIndex() ? -1 : quad.colorIndex();
		final boolean emissive = mat.emissive();
		final class_4588 vertexConsumer = getVertexConsumer(mat.blendMode(), mat.glint());

		colorizeQuad(quad, colorIndex);
		shadeQuad(quad, emissive);
		bufferQuad(quad, vertexConsumer);
	}

	private void colorizeQuad(MutableQuadViewImpl quad, int colorIndex) {
		if (colorIndex != -1) {
			final int itemColor = colorMap.method_1704(itemStack, colorIndex);

			for (int i = 0; i < 4; i++) {
				quad.color(i, ColorHelper.multiplyColor(itemColor, quad.color(i)));
			}
		}
	}

	private void shadeQuad(MutableQuadViewImpl quad, boolean emissive) {
		if (emissive) {
			for (int i = 0; i < 4; i++) {
				quad.lightmap(i, class_765.field_32767);
			}
		} else {
			final int lightmap = this.lightmap;

			for (int i = 0; i < 4; i++) {
				quad.lightmap(i, ColorHelper.maxBrightness(quad.lightmap(i), lightmap));
			}
		}
	}

	/**
	 * Caches custom blend mode / vertex consumers and mimics the logic
	 * in {@code RenderLayers.getItemLayer}. Layers other than
	 * translucent are mapped to cutout.
	 */
	private class_4588 getVertexConsumer(BlendMode blendMode, TriState glintMode) {
		boolean translucent;
		boolean glint;

		if (blendMode == BlendMode.DEFAULT) {
			translucent = isDefaultTranslucent;
		} else {
			translucent = blendMode == BlendMode.TRANSLUCENT;
		}

		if (glintMode == TriState.DEFAULT) {
			glint = isDefaultGlint;
		} else {
			glint = glintMode == TriState.TRUE;
		}

		if (translucent) {
			if (glint) {
				if (translucentGlintVertexConsumer == null) {
					translucentGlintVertexConsumer = createVertexConsumer(class_4722.method_29382(), true);
				}

				return translucentGlintVertexConsumer;
			} else {
				if (translucentVertexConsumer == null) {
					translucentVertexConsumer = createVertexConsumer(class_4722.method_29382(), false);
				}

				return translucentVertexConsumer;
			}
		} else {
			if (glint) {
				if (cutoutGlintVertexConsumer == null) {
					cutoutGlintVertexConsumer = createVertexConsumer(class_4722.method_24074(), true);
				}

				return cutoutGlintVertexConsumer;
			} else {
				if (cutoutVertexConsumer == null) {
					cutoutVertexConsumer = createVertexConsumer(class_4722.method_24074(), false);
				}

				return cutoutVertexConsumer;
			}
		}
	}

	private class_4588 createVertexConsumer(class_1921 layer, boolean glint) {
		if (isGlintDynamicDisplay && glint) {
			if (dynamicDisplayGlintEntry == null) {
				dynamicDisplayGlintEntry = matrixStack.method_23760().method_56822();

				if (transformMode == class_811.field_4317) {
					class_7837.method_46414(dynamicDisplayGlintEntry.method_23761(), 0.5F);
				} else if (transformMode.method_29998()) {
					class_7837.method_46414(dynamicDisplayGlintEntry.method_23761(), 0.75F);
				}
			}

			return class_918.method_30114(vertexConsumerProvider, layer, dynamicDisplayGlintEntry);
		}

		return class_918.method_23181(vertexConsumerProvider, layer, true, glint);
	}
}
