/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.mixin.resource.conditions;

import java.io.IOException;
import java.io.Reader;
import java.util.Map;

import com.google.gson.JsonElement;
import com.llamalad7.mixinextras.sugar.Local;
import com.mojang.serialization.Decoder;
import org.spongepowered.asm.mixin.Mixin;
import org.spongepowered.asm.mixin.Unique;
import org.spongepowered.asm.mixin.injection.At;
import org.spongepowered.asm.mixin.injection.Inject;
import org.spongepowered.asm.mixin.injection.callback.CallbackInfo;
import net.fabricmc.fabric.impl.resource.conditions.ResourceConditionsImpl;
import net.minecraft.class_2385;
import net.minecraft.class_3298;
import net.minecraft.class_3300;
import net.minecraft.class_5321;
import net.minecraft.class_6903;
import net.minecraft.class_7655;
import net.minecraft.class_9248;

@Mixin(class_7655.class)
public class RegistryLoaderMixin {
	@Unique
	private static final ThreadLocal<class_6903.class_7863> REGISTRY_INFO = new ThreadLocal<>();

	/**
	 * Capture the current registries, so they can be passed to the resource conditions.
	 */
	@Inject(method = "loadFromResource(Lnet/minecraft/resource/ResourceManager;Lnet/minecraft/registry/RegistryOps$RegistryInfoGetter;Lnet/minecraft/registry/MutableRegistry;Lcom/mojang/serialization/Decoder;Ljava/util/Map;)V", at = @At("HEAD"))
	private static <E> void captureRegistries(class_3300 resourceManager, class_6903.class_7863 infoGetter, class_2385<E> registry, Decoder<E> elementDecoder, Map<class_5321<?>, Exception> errors, CallbackInfo ci) {
		REGISTRY_INFO.set(infoGetter);
	}

	@Inject(method = "loadFromResource(Lnet/minecraft/resource/ResourceManager;Lnet/minecraft/registry/RegistryOps$RegistryInfoGetter;Lnet/minecraft/registry/MutableRegistry;Lcom/mojang/serialization/Decoder;Ljava/util/Map;)V", at = @At("RETURN"))
	private static <E> void releaseRegistries(class_3300 resourceManager, class_6903.class_7863 infoGetter, class_2385<E> registry, Decoder<E> elementDecoder, Map<class_5321<?>, Exception> errors, CallbackInfo ci) {
		REGISTRY_INFO.remove();
	}

	@Inject(
			method = "Lnet/minecraft/registry/RegistryLoader;parseAndAdd(Lnet/minecraft/registry/MutableRegistry;Lcom/mojang/serialization/Decoder;Lnet/minecraft/registry/RegistryOps;Lnet/minecraft/registry/RegistryKey;Lnet/minecraft/resource/Resource;Lnet/minecraft/registry/entry/RegistryEntryInfo;)V",
			at = @At(value = "INVOKE", target = "Lcom/mojang/serialization/Decoder;parse(Lcom/mojang/serialization/DynamicOps;Ljava/lang/Object;)Lcom/mojang/serialization/DataResult;", remap = false),
			cancellable = true
	)
	private static <E> void checkResourceCondition(
			class_2385<E> registry, Decoder<E> decoder, class_6903<JsonElement> ops, class_5321<E> key, class_3298 resource, class_9248 entryInfo,
			CallbackInfo ci, @Local Reader reader, @Local JsonElement jsonElement
	) throws IOException {
		// This method is called both on the server (when loading resources) and on the client (when syncing from the
		// server). We only want to apply resource conditions when loading via loadFromResource.
		class_6903.class_7863 registryInfoGetter = REGISTRY_INFO.get();
		if (registryInfoGetter == null) return;

		if (jsonElement.isJsonObject() && !ResourceConditionsImpl.applyResourceConditions(jsonElement.getAsJsonObject(), key.method_41185().toString(), key.method_29177(), registryInfoGetter)) {
			reader.close();
			ci.cancel();
		}
	}
}
