/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.impl.resource.loader;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.function.Consumer;
import java.util.function.Predicate;

import org.jetbrains.annotations.Nullable;
import org.jetbrains.annotations.VisibleForTesting;
import net.fabricmc.fabric.api.resource.ModResourcePack;
import net.minecraft.class_2561;
import net.minecraft.class_3264;
import net.minecraft.class_3285;
import net.minecraft.class_3288;
import net.minecraft.class_5352;
import net.minecraft.class_9224;
import net.minecraft.class_9225;

/**
 * Represents a resource pack provider for mods and built-in mods resource packs.
 */
public class ModResourcePackCreator implements class_3285 {
	/**
	 * The ID of the root resource pack profile for bundled packs.
	 */
	public static final String FABRIC = "fabric";
	private static final String PROGRAMMER_ART = "programmer_art";
	private static final String HIGH_CONTRAST = "high_contrast";
	public static final Set<String> POST_CHANGE_HANDLE_REQUIRED = Set.of(FABRIC, PROGRAMMER_ART, HIGH_CONTRAST);
	@VisibleForTesting
	public static final Predicate<Set<String>> BASE_PARENT = enabled -> enabled.contains(FABRIC);
	@VisibleForTesting
	public static final Predicate<Set<String>> PROGRAMMER_ART_PARENT = enabled -> enabled.contains(FABRIC) && enabled.contains(PROGRAMMER_ART);
	@VisibleForTesting
	public static final Predicate<Set<String>> HIGH_CONTRAST_PARENT = enabled -> enabled.contains(FABRIC) && enabled.contains(HIGH_CONTRAST);
	/**
	 * This can be used to check if a pack profile is for mod-provided packs.
	 */
	public static final class_5352 RESOURCE_PACK_SOURCE = new class_5352() {
		@Override
		public class_2561 method_45282(class_2561 packName) {
			return class_2561.method_43469("pack.nameAndSource", packName, class_2561.method_43471("pack.source.fabricmod"));
		}

		@Override
		public boolean method_45279() {
			return true;
		}
	};
	public static final ModResourcePackCreator CLIENT_RESOURCE_PACK_PROVIDER = new ModResourcePackCreator(class_3264.field_14188);
	/**
	 * The maximum number of known data packs requested from the client, including vanilla data packs.
	 */
	public static final int MAX_KNOWN_PACKS = Integer.getInteger("fabric-resource-loader-v0:maxKnownPacks", 1024);

	private final class_3264 type;
	private final class_9225 activationInfo;
	private final boolean forClientDataPackManager;

	public ModResourcePackCreator(class_3264 type) {
		this(type, false);
	}

	protected ModResourcePackCreator(class_3264 type, boolean forClientDataPackManager) {
		this.type = type;
		this.activationInfo = new class_9225(!forClientDataPackManager, class_3288.class_3289.field_14280, false);
		this.forClientDataPackManager = forClientDataPackManager;
	}

	/**
	 * Registers the resource packs.
	 *
	 * @param consumer The resource pack profile consumer.
	 */
	@Override
	public void method_14453(Consumer<class_3288> consumer) {
		/*
			Register order rule in this provider:
			1. Mod resource packs
			2. Mod built-in resource packs

			Register order rule globally:
			1. Default and Vanilla built-in resource packs
			2. Mod resource packs
			3. Mod built-in resource packs
			4. User resource packs
		 */

		class_9224 metadata = new class_9224(
				FABRIC,
				class_2561.method_43471("pack.name.fabricMods"),
				RESOURCE_PACK_SOURCE,
				Optional.empty()
		);

		consumer.accept(class_3288.method_45275(
				metadata,
				new PlaceholderResourcePack.Factory(this.type, metadata),
				this.type,
				this.activationInfo
		));

		// Build a list of mod resource packs.
		registerModPack(consumer, null, BASE_PARENT);

		if (this.type == class_3264.field_14188) {
			// Programmer Art/High Contrast data packs can never be enabled.
			registerModPack(consumer, PROGRAMMER_ART, PROGRAMMER_ART_PARENT);
			registerModPack(consumer, HIGH_CONTRAST, HIGH_CONTRAST_PARENT);
		}

		// Register all built-in resource packs provided by mods.
		ResourceManagerHelperImpl.registerBuiltinResourcePacks(this.type, consumer);
	}

	private void registerModPack(Consumer<class_3288> consumer, @Nullable String subPath, Predicate<Set<String>> parents) {
		List<ModResourcePack> packs = new ArrayList<>();
		ModResourcePackUtil.appendModResourcePacks(packs, this.type, subPath);

		for (ModResourcePack pack : packs) {
			class_3288 profile = class_3288.method_45275(
					pack.method_56926(),
					new ModResourcePackFactory(pack),
					this.type,
					this.activationInfo
			);

			if (profile != null) {
				if (!forClientDataPackManager) {
					((FabricResourcePackProfile) profile).fabric_setParentsPredicate(parents);
				}

				consumer.accept(profile);
			}
		}
	}
}
