/*
 * Decompiled with CFR 0.152.
 */
package net.fabricmc.loom.util.download;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.UncheckedIOException;
import java.net.ProxySelector;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.http.HttpClient;
import java.net.http.HttpHeaders;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.nio.charset.StandardCharsets;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.attribute.BasicFileAttributeView;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.FileTime;
import java.time.Duration;
import java.time.Instant;
import java.util.Locale;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.IntConsumer;
import java.util.zip.GZIPInputStream;
import net.fabricmc.loom.util.AttributeHelper;
import net.fabricmc.loom.util.Checksum;
import net.fabricmc.loom.util.download.DownloadBuilder;
import net.fabricmc.loom.util.download.DownloadException;
import net.fabricmc.loom.util.download.DownloadProgressListener;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Download {
    private static final String E_TAG = "ETag";
    private static final Logger LOGGER = LoggerFactory.getLogger(Download.class);
    private final URI url;
    private final String expectedHash;
    private final boolean useEtag;
    private final boolean forceDownload;
    private final boolean offline;
    private final Duration maxAge;
    private final DownloadProgressListener progressListener;

    public static DownloadBuilder create(String url) throws URISyntaxException {
        return DownloadBuilder.create(url);
    }

    Download(URI url, String expectedHash, boolean useEtag, boolean forceDownload, boolean offline, Duration maxAge, DownloadProgressListener progressListener) {
        this.url = url;
        this.expectedHash = expectedHash;
        this.useEtag = useEtag;
        this.forceDownload = forceDownload;
        this.offline = offline;
        this.maxAge = maxAge;
        this.progressListener = progressListener;
    }

    private HttpClient getHttpClient() throws DownloadException {
        if (this.offline) {
            throw this.error("Unable to download %s in offline mode", this.url);
        }
        return HttpClient.newBuilder().followRedirects(HttpClient.Redirect.ALWAYS).proxy(ProxySelector.getDefault()).build();
    }

    private HttpRequest getRequest() {
        return HttpRequest.newBuilder(this.url).GET().build();
    }

    private HttpRequest getETagRequest(String etag) {
        return HttpRequest.newBuilder(this.url).GET().header("If-None-Match", etag).build();
    }

    private <T> HttpResponse<T> send(HttpRequest httpRequest, HttpResponse.BodyHandler<T> bodyHandler) throws DownloadException {
        this.progressListener.onStart();
        try {
            return this.getHttpClient().send(httpRequest, bodyHandler);
        }
        catch (IOException | InterruptedException e) {
            throw this.error(e, "Failed to download (%s)", this.url);
        }
    }

    String downloadString() throws DownloadException {
        boolean successful;
        HttpResponse<InputStream> response = this.send(this.getRequest(), HttpResponse.BodyHandlers.ofInputStream());
        int statusCode = response.statusCode();
        boolean bl = successful = statusCode >= 200 && statusCode < 300;
        if (!successful) {
            throw this.error("HTTP request to (%s) returned unsuccessful status (%d)", this.url, statusCode);
        }
        try {
            String string;
            block12: {
                InputStream inputStream = this.decodeOutput(response);
                try {
                    string = new String(inputStream.readAllBytes(), StandardCharsets.UTF_8);
                    if (inputStream == null) break block12;
                }
                catch (Throwable throwable) {
                    try {
                        if (inputStream != null) {
                            try {
                                inputStream.close();
                            }
                            catch (Throwable throwable2) {
                                throwable.addSuppressed(throwable2);
                            }
                        }
                        throw throwable;
                    }
                    catch (IOException e) {
                        throw this.error(e, "Failed to decode download output", new Object[0]);
                    }
                }
                inputStream.close();
            }
            return string;
        }
        finally {
            this.progressListener.onEnd();
        }
    }

    void downloadPath(Path output) throws DownloadException {
        boolean downloadRequired = this.requiresDownload(output);
        if (!downloadRequired) {
            return;
        }
        try {
            this.doDownload(output);
        }
        catch (Throwable throwable) {
            this.tryCleanup(output);
            throw this.error(throwable, "Failed to download (%s) to (%s)", this.url, output);
        }
        finally {
            this.progressListener.onEnd();
        }
    }

    private void doDownload(Path output) throws DownloadException {
        HttpHeaders headers;
        String responseETag;
        HttpResponse<InputStream> response;
        block18: {
            boolean success;
            Optional<Object> eTag = Optional.empty();
            if (!this.forceDownload && this.useEtag && Download.exists(output)) {
                eTag = this.readEtag(output);
            }
            try {
                Files.createDirectories(output.getParent(), new FileAttribute[0]);
                Files.deleteIfExists(output);
            }
            catch (IOException e) {
                throw this.error(e, "Failed to prepare path for download", new Object[0]);
            }
            HttpRequest httpRequest = eTag.map(this::getETagRequest).orElseGet(this::getRequest);
            this.createLock(output);
            response = this.send(httpRequest, HttpResponse.BodyHandlers.ofInputStream());
            this.getAndResetLock(output);
            int statusCode = response.statusCode();
            boolean bl = success = statusCode == 304 || statusCode >= 200 && statusCode < 300;
            if (statusCode == 304) {
                return;
            }
            if (success) {
                long length = Long.parseLong(response.headers().firstValue("Content-Length").orElse("-1"));
                AtomicLong totalBytes = new AtomicLong(0L);
                try (OutputStream outputStream = Files.newOutputStream(output, new OpenOption[0]);){
                    this.copyWithCallback(this.decodeOutput(response), outputStream, value -> {
                        if (length < 0L) {
                            return;
                        }
                        this.progressListener.onProgress(totalBytes.addAndGet(value), length);
                    });
                    break block18;
                }
                catch (IOException e) {
                    this.tryCleanup(output);
                    throw this.error(e, "Failed to decode and write download output", new Object[0]);
                }
            }
            this.tryCleanup(output);
            throw this.error("HTTP request to (%s) returned unsuccessful status (%d)", this.url, statusCode);
        }
        if (this.useEtag && (responseETag = (String)(headers = response.headers()).firstValue(E_TAG.toLowerCase(Locale.ROOT)).orElse(null)) != null) {
            this.writeEtag(output, responseETag);
        }
        if (this.expectedHash != null) {
            if (!this.isHashValid(output)) {
                String downloadedHash;
                try {
                    downloadedHash = Checksum.sha1Hex(output);
                    Files.deleteIfExists(output);
                }
                catch (IOException e) {
                    downloadedHash = "unknown hash";
                }
                throw this.error("Failed to download (%s) with expected hash: %s got %s", this.url, this.expectedHash, downloadedHash);
            }
            this.writeHash(output, this.expectedHash);
        }
    }

    private void copyWithCallback(InputStream is, OutputStream os, IntConsumer consumer) throws IOException {
        int length;
        byte[] buffer = new byte[1024];
        while ((length = is.read(buffer)) > 0) {
            os.write(buffer, 0, length);
            consumer.accept(length);
        }
    }

    private InputStream decodeOutput(HttpResponse<InputStream> response) throws IOException {
        String encoding;
        return switch (encoding = response.headers().firstValue("Content-Encoding").orElse("")) {
            case "gzip" -> new GZIPInputStream(response.body());
            case "" -> response.body();
            default -> throw this.error("Unsupported encoding: %s", encoding);
        };
    }

    private boolean requiresDownload(Path output) throws DownloadException {
        if (this.getAndResetLock(output)) {
            LOGGER.warn("Forcing downloading {} as existing lock file was found. This may happen if the gradle build was forcefully canceled.", (Object)output);
            return true;
        }
        if (this.forceDownload || !Download.exists(output)) {
            return true;
        }
        if (this.offline) {
            return false;
        }
        if (this.expectedHash != null) {
            String hashAttribute = this.readHash(output).orElse("");
            if (this.expectedHash.equalsIgnoreCase(hashAttribute)) {
                return false;
            }
            if (this.isHashValid(output)) {
                this.writeHash(output, this.expectedHash);
                return false;
            }
            if (System.getProperty("fabric.loom.test") != null) {
                throw this.error("Download file (%s) may have been modified", output);
            }
            LOGGER.info("Found existing file ({}) to download with unexpected hash.", (Object)output);
        }
        return this.maxAge.equals(Duration.ZERO) || this.isOutdated(output);
    }

    private boolean isHashValid(Path path) {
        int i = this.expectedHash.indexOf(58);
        String algorithm = this.expectedHash.substring(0, i);
        String hash = this.expectedHash.substring(i + 1);
        try {
            switch (algorithm) {
                case "sha1": {
                    break;
                }
                default: {
                    throw this.error("Unsupported hash algorithm (%s)", algorithm);
                }
            }
            String computedHash = Checksum.sha1Hex(path);
            return computedHash.equalsIgnoreCase(hash);
        }
        catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }

    private boolean isOutdated(Path path) throws DownloadException {
        try {
            FileTime lastModified = this.getLastModified(path);
            return lastModified.toInstant().plus(this.maxAge).isBefore(Instant.now());
        }
        catch (IOException e) {
            throw this.error(e, "Failed to check if (%s) is outdated", path);
        }
    }

    private Optional<String> readEtag(Path output) {
        try {
            return AttributeHelper.readAttribute(output, E_TAG);
        }
        catch (IOException e) {
            return Optional.empty();
        }
    }

    private void writeEtag(Path output, String eTag) throws DownloadException {
        try {
            AttributeHelper.writeAttribute(output, E_TAG, eTag);
        }
        catch (IOException e) {
            throw this.error(e, "Failed to write etag to (%s)", output);
        }
    }

    private Optional<String> readHash(Path output) {
        try {
            return AttributeHelper.readAttribute(output, "LoomHash");
        }
        catch (IOException e) {
            return Optional.empty();
        }
    }

    private void writeHash(Path output, String value) throws DownloadException {
        try {
            AttributeHelper.writeAttribute(output, "LoomHash", value);
        }
        catch (IOException e) {
            throw this.error(e, "Failed to write hash to (%s)", output);
        }
    }

    private void tryCleanup(Path output) {
        try {
            Files.deleteIfExists(output);
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    private static boolean exists(Path path) {
        return path.getFileSystem() == FileSystems.getDefault() ? path.toFile().exists() : Files.exists(path, new LinkOption[0]);
    }

    private FileTime getLastModified(Path path) throws IOException {
        BasicFileAttributeView basicView = Files.getFileAttributeView(path, BasicFileAttributeView.class, new LinkOption[0]);
        return basicView.readAttributes().lastModifiedTime();
    }

    private Path getLockFile(Path output) {
        return output.resolveSibling(output.getFileName() + ".lock");
    }

    private boolean getAndResetLock(Path output) throws DownloadException {
        Path lock = this.getLockFile(output);
        boolean exists = Download.exists(lock);
        if (exists) {
            try {
                Files.delete(lock);
            }
            catch (IOException e) {
                throw this.error(e, "Failed to release lock on %s", lock);
            }
        }
        return exists;
    }

    private void createLock(Path output) throws DownloadException {
        Path lock = this.getLockFile(output);
        try {
            Files.createFile(lock, new FileAttribute[0]);
        }
        catch (IOException e) {
            throw this.error(e, "Failed to acquire lock on %s", lock);
        }
    }

    private DownloadException error(String message, Object ... args) {
        return new DownloadException(String.format(Locale.ENGLISH, message, args));
    }

    private DownloadException error(Throwable throwable) {
        return new DownloadException(throwable);
    }

    private DownloadException error(Throwable throwable, String message, Object ... args) {
        return new DownloadException(message.formatted(args), throwable);
    }
}

