/*
 * Copyright 2020 zml
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package ca.stellardrift.confabricate;

import com.google.common.collect.ImmutableSet;
import org.checkerframework.checker.nullness.qual.NonNull;
import org.spongepowered.configurate.BasicConfigurationNode;
import org.spongepowered.configurate.ConfigurationNode;
import org.spongepowered.configurate.ConfigurationOptions;

import java.io.IOException;
import java.util.List;
import java.util.Map;
import net.minecraft.class_2479;
import net.minecraft.class_2481;
import net.minecraft.class_2487;
import net.minecraft.class_2489;
import net.minecraft.class_2491;
import net.minecraft.class_2494;
import net.minecraft.class_2495;
import net.minecraft.class_2497;
import net.minecraft.class_2499;
import net.minecraft.class_2501;
import net.minecraft.class_2503;
import net.minecraft.class_2516;
import net.minecraft.class_2519;
import net.minecraft.class_2520;

/**
 * A configuration adapter that will convert Minecraft NBT data into a
 * Configurate {@link ConfigurationNode}.
 *
 * @since 1.0.0
 */
public final class NbtNodeAdapter {

    private NbtNodeAdapter() {}

    /**
     * Given a tag, convert it to a node.
     *
     * <p>Depending on the configuration of the provided node, the conversion
     * may lose some data when roundtripped back. For example, array tags may
     * be converted to lists if the node provided does not support arrays.
     *
     * @param tag the tag to convert
     * @param node the node to populate
     * @throws IOException if invalid tags are provided
     * @since 1.0.0
     */
    public static void tagToNode(final class_2520 tag, final ConfigurationNode node) throws IOException {
        if (tag instanceof class_2487) {
            final class_2487 compoundTag = (class_2487) tag;
            for (String key : compoundTag.method_10541()) {
                tagToNode(compoundTag.method_10580(key), node.node(key));
            }
        } else if (tag instanceof class_2499) {
            for (class_2520 value : (class_2499) tag) {
                tagToNode(value, node.appendListNode());
            }
        } else if (tag instanceof class_2519) {
            node.raw(tag.method_10714());
        } else if (tag instanceof class_2481) {
            node.raw(((class_2481) tag).method_10698());
        } else if (tag instanceof class_2516) {
            node.raw(((class_2516) tag).method_10696());
        } else if (tag instanceof class_2497) {
            node.raw(((class_2497) tag).method_10701());
        } else if (tag instanceof class_2503) {
            node.raw(((class_2503) tag).method_10699());
        } else if (tag instanceof class_2494) {
            node.raw(((class_2494) tag).method_10700());
        } else if (tag instanceof class_2489) {
            node.raw(((class_2489) tag).method_10697());
        } else if (tag instanceof class_2479) {
            if (node.options().acceptsType(byte[].class)) {
                node.raw(((class_2479) tag).method_10521());
            } else {
                node.raw(null);
                for (byte b : ((class_2479) tag).method_10521()) {
                    node.appendListNode().raw(b);
                }
            }
        } else if (tag instanceof class_2495) {
            if (node.options().acceptsType(int[].class)) {
                node.raw(((class_2495) tag).method_10588());
            } else {
                node.raw(null);
                for (int i : ((class_2495) tag).method_10588()) {
                    node.appendListNode().raw(i);
                }
            }

        } else if (tag instanceof class_2501) {
            if (node.options().acceptsType(long[].class)) {
                node.raw(((class_2501) tag).method_10615());
            } else {
                node.raw(null);
                for (long l : ((class_2501) tag).method_10615()) {
                    node.appendListNode().raw(l);
                }
            }
        } else if (tag instanceof class_2491) {
            // no-op
        } else {
            throw new IOException("Unknown tag type: " + tag.getClass());
        }
    }

    /**
     * Convert a node to tag. Because NBT is strongly typed and does not permit
     * lists with mixed types, some configuration nodes will not be convertible
     * to Tags.
     *
     * @param node the configuration node
     * @return the converted tag object
     * @throws IOException if an IO error occurs while converting the tag
     * @since 1.0.0
     */
    public static class_2520 nodeToTag(final ConfigurationNode node) throws IOException {
        if (node.isMap()) {
            final class_2487 tag = new class_2487();
            for (Map.Entry<Object, ? extends ConfigurationNode> ent : node.childrenMap().entrySet()) {
                tag.method_10566(ent.getKey().toString(), nodeToTag(ent.getValue()));
            }
            return tag;
        } else if (node.isList()) {
            final class_2499 list = new class_2499();
            for (ConfigurationNode child : node.childrenList()) {
                list.add(nodeToTag(child));
            }
            return list;
        } else {
            final Object obj = node.raw();
            if (obj instanceof byte[]) {
                return new class_2479((byte[]) obj);
            } else if (obj instanceof int[]) {
                return new class_2495((int[]) obj);
            } else if (obj instanceof long[]) {
                return new class_2501((long[]) obj);
            } else if (obj instanceof Byte) {
                return class_2481.method_23233((Byte) obj);
            } else if (obj instanceof Short) {
                return class_2516.method_23254((Short) obj);
            } else if (obj instanceof Integer) {
                return class_2497.method_23247((Integer) obj);
            } else if (obj instanceof Long) {
                return class_2503.method_23251((Long) obj);
            } else if (obj instanceof Float) {
                return class_2494.method_23244((Float) obj);
            } else if (obj instanceof Double) {
                return class_2489.method_23241((Double) obj);
            } else if (obj instanceof String) {
                return class_2519.method_23256((String) obj);
            } else {
                throw new IOException("Unsupported object type " + (obj == null ? null : obj.getClass()));
            }
        }
    }

    /**
     * Create an empty node with options appropriate for handling NBT data.
     *
     * @return the new node
     * @since 1.0.0
     */
    public static ConfigurationNode createEmptyNode() {
        return createEmptyNode(Confabricate.confabricateOptions());
    }

    /**
     * Create an empty node with options appropriate for handling NBT data.
     *
     * @param options options to work with
     * @return the new node
     * @since 1.0.0
     */
    public static ConfigurationNode createEmptyNode(final @NonNull ConfigurationOptions options) {
        return BasicConfigurationNode.root(options
                .nativeTypes(ImmutableSet.of(Map.class, List.class, Byte.class,
                        Short.class, Integer.class, Long.class, Float.class, Double.class,
                        long[].class, byte[].class, int[].class, String.class)));
    }

}
