/*
 * Copyright 2020 zml
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package ca.stellardrift.confabricate;

import java.io.IOException;
import java.util.List;
import java.util.Map;
import java.util.Set;
import net.minecraft.class_2479;
import net.minecraft.class_2481;
import net.minecraft.class_2487;
import net.minecraft.class_2489;
import net.minecraft.class_2491;
import net.minecraft.class_2494;
import net.minecraft.class_2495;
import net.minecraft.class_2497;
import net.minecraft.class_2499;
import net.minecraft.class_2501;
import net.minecraft.class_2503;
import net.minecraft.class_2516;
import net.minecraft.class_2519;
import net.minecraft.class_2520;
import org.checkerframework.checker.nullness.qual.NonNull;
import org.spongepowered.configurate.BasicConfigurationNode;
import org.spongepowered.configurate.ConfigurationNode;
import org.spongepowered.configurate.ConfigurationNodeFactory;
import org.spongepowered.configurate.ConfigurationOptions;

/**
 * A configuration adapter that will convert Minecraft NBT data into a
 * Configurate {@link ConfigurationNode}.
 *
 * @since 1.0.0
 */
public final class NbtNodeAdapter {

    private static final ConfigurationNodeFactory<BasicConfigurationNode> FACTORY = new ConfigurationNodeFactory<>() {

        @Override
        public BasicConfigurationNode createNode(final ConfigurationOptions options) {
            return BasicConfigurationNode.root(options
                    .nativeTypes(Set.of(Map.class, List.class, Byte.class,
                            Short.class, Integer.class, Long.class, Float.class, Double.class,
                            long[].class, byte[].class, int[].class, String.class)));
        }

    };

    private NbtNodeAdapter() {}

    /**
     * Given a tag, convert it to a node.
     *
     * <p>Depending on the configuration of the provided node, the conversion
     * may lose some data when roundtripped back. For example, array tags may
     * be converted to lists if the node provided does not support arrays.
     *
     * @param tag the tag to convert
     * @param node the node to populate
     * @throws IOException if invalid tags are provided
     * @since 1.0.0
     */
    public static void tagToNode(final class_2520 tag, final ConfigurationNode node) throws IOException {
        if (tag instanceof final class_2487 compoundTag) {
            for (final String key : compoundTag.method_10541()) {
                tagToNode(compoundTag.method_10580(key), node.node(key));
            }
        } else if (tag instanceof final class_2499 list) {
            for (final class_2520 value : list) {
                tagToNode(value, node.appendListNode());
            }
        } else if (tag instanceof class_2519) {
            node.raw(tag.method_10714());
        } else if (tag instanceof final class_2481 b) {
            node.raw(b.method_10698());
        } else if (tag instanceof final class_2516 s) {
            node.raw(s.method_10696());
        } else if (tag instanceof final class_2497 i) {
            node.raw(i.method_10701());
        } else if (tag instanceof final class_2503 l) {
            node.raw(l.method_10699());
        } else if (tag instanceof final class_2494 f) {
            node.raw(f.method_10700());
        } else if (tag instanceof final class_2489 d) {
            node.raw(d.method_10697());
        } else if (tag instanceof final class_2479 arr) {
            if (node.options().acceptsType(byte[].class)) {
                node.raw(arr.method_10521());
            } else {
                node.raw(null);
                for (final byte b : arr.method_10521()) {
                    node.appendListNode().raw(b);
                }
            }
        } else if (tag instanceof final class_2495 arr) {
            if (node.options().acceptsType(int[].class)) {
                node.raw(arr.method_10588());
            } else {
                node.raw(null);
                for (final int i : arr.method_10588()) {
                    node.appendListNode().raw(i);
                }
            }

        } else if (tag instanceof final class_2501 arr) {
            if (node.options().acceptsType(long[].class)) {
                node.raw(arr.method_10615());
            } else {
                node.raw(null);
                for (final long l : arr.method_10615()) {
                    node.appendListNode().raw(l);
                }
            }
        } else if (tag instanceof class_2491) {
            // no-op
        } else {
            throw new IOException("Unknown tag type: " + tag.getClass());
        }
    }

    /**
     * Convert a node to tag. Because NBT is strongly typed and does not permit
     * lists with mixed types, some configuration nodes will not be convertible
     * to Tags.
     *
     * @param node the configuration node
     * @return the converted tag object
     * @throws IOException if an IO error occurs while converting the tag
     * @since 1.0.0
     */
    public static class_2520 nodeToTag(final ConfigurationNode node) throws IOException {
        if (node.isMap()) {
            final class_2487 tag = new class_2487();
            for (final Map.Entry<Object, ? extends ConfigurationNode> ent : node.childrenMap().entrySet()) {
                tag.method_10566(ent.getKey().toString(), nodeToTag(ent.getValue()));
            }
            return tag;
        } else if (node.isList()) {
            final class_2499 list = new class_2499();
            for (final ConfigurationNode child : node.childrenList()) {
                list.add(nodeToTag(child));
            }
            return list;
        } else {
            final Object obj = node.raw();
            if (obj instanceof final byte[] arr) {
                return new class_2479(arr);
            } else if (obj instanceof final int[] arr) {
                return new class_2495(arr);
            } else if (obj instanceof final long[] arr) {
                return new class_2501(arr);
            } else if (obj instanceof final Byte b) {
                return class_2481.method_23233(b);
            } else if (obj instanceof final Short s) {
                return class_2516.method_23254(s);
            } else if (obj instanceof final Integer i) {
                return class_2497.method_23247(i);
            } else if (obj instanceof final Long l) {
                return class_2503.method_23251(l);
            } else if (obj instanceof final Float f) {
                return class_2494.method_23244(f);
            } else if (obj instanceof final Double d) {
                return class_2489.method_23241(d);
            } else if (obj instanceof final String s) {
                return class_2519.method_23256(s);
            } else {
                throw new IOException("Unsupported object type " + (obj == null ? null : obj.getClass()));
            }
        }
    }

    /**
     * Create an empty node with options appropriate for handling NBT data.
     *
     * @return the new node
     * @since 1.0.0
     */
    public static ConfigurationNode createEmptyNode() {
        return FACTORY.createNode(Confabricate.confabricateOptions());
    }

    /**
     * Create an empty node with options appropriate for handling NBT data.
     *
     * @param options options to work with
     * @return the new node
     * @since 1.0.0
     */
    public static ConfigurationNode createEmptyNode(final @NonNull ConfigurationOptions options) {
        return FACTORY.createNode(options);
    }

    /**
     * Get a factory for nodes prepared to handle NBT data.
     *
     * @return the factory
     * @since 3.0.0
     */
    public static ConfigurationNodeFactory<BasicConfigurationNode> nodeFactory() {
        return FACTORY;
    }

}
