/*
 * Copyright 2020 zml
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package ca.stellardrift.confabricate.typeserializers;

import com.google.common.collect.ImmutableList;
import com.google.errorprone.annotations.concurrent.LazyInit;
import com.mojang.datafixers.util.Either;
import java.util.List;
import java.util.function.Consumer;
import java.util.function.Supplier;
import net.minecraft.class_151;
import net.minecraft.class_2378;
import net.minecraft.class_2960;
import net.minecraft.class_5321;
import net.minecraft.class_6862;
import net.minecraft.class_6880;
import net.minecraft.class_6885;
import org.spongepowered.configurate.ConfigurationNode;
import org.spongepowered.configurate.serialize.SerializationException;

/**
 * A tag implementation that keeps its original form for reserialization.
 *
 * @param <V> element type
 */
final class ConfabricateHolderSet<V> extends class_6885.class_6887<V> {

    private final List<TagEntry<V>> serializedForm;
    private final Supplier<class_2378<V>> elementResolver;
    @LazyInit private volatile List<class_6880<V>> values;

    /**
     * Create a new lazily initialized tag.
     *
     * @param serializedForm serialized form of the tag
     * @param elementResolver element-based resolver
     */
    ConfabricateHolderSet(final List<TagEntry<V>> serializedForm, final Supplier<class_2378<V>> elementResolver) {
        this.serializedForm = List.copyOf(serializedForm);
        this.elementResolver = elementResolver;
    }

    public List<TagEntry<V>> serializedForm() {
        return this.serializedForm;
    }

    private List<class_6880<V>> resolve() {
        final ImmutableList.Builder<class_6880<V>> builder = ImmutableList.builder();

        final class_2378<V> registry = this.elementResolver.get();
        for (final TagEntry<V> entry : this.serializedForm) {
            entry.collect(registry, builder::add);
        }

        return builder.build();
    }

    @Override
    public List<class_6880<V>> method_40249() {
        List<class_6880<V>> values = this.values;
        if (values == null) {
            this.values = values = this.resolve();
        }
        return values;
    }

    @Override
    public Either<class_6862<V>, List<class_6880<V>>> method_40248() {
        return Either.right(this.method_40249());
    }

    @Override
    public boolean method_40241(final class_6880<V> entry) {
        return this.method_40249().contains(entry);
    }

    sealed interface TagEntry<V> {

        String TAG_PREFIX = "#";
        String ID = "id";
        String REQUIRED = "required";

        static <E> TagEntry<E> fromNode(
            final class_5321<? extends class_2378<E>> registry,
            final ConfigurationNode value
        ) throws SerializationException {
            final String id;
            final boolean required;
            if (value.isMap()) { // reference to optional tag
                id = value.node(ID).getString();
                required = value.node(REQUIRED).getBoolean();
            } else {
                id = value.getString();
                required = true;
            }

            if (id == null) {
                throw new SerializationException("a tag id field is required to deserialize");
            }

            try {
                if (id.startsWith(TAG_PREFIX)) {
                    final class_2960 loc = new class_2960(id.substring(1));
                    return new Tag<>(class_6862.method_40092(registry, loc), required);
                    // return required ? new Tag.TagEntry(ident) : new Tag.OptionalTagEntry(ident);
                } else {
                    final class_2960 loc = new class_2960(id);
                    return new Single<>(class_5321.method_29179(registry, loc), required);
                }
            } catch (final class_151 ex) {
                throw new SerializationException("Invalid resource location " + id);
            }
        }

        boolean required();

        void collect(class_2378<V> registry, Consumer<class_6880<V>> collector);

        void toNode(ConfigurationNode target) throws SerializationException;

        record Single<V>(class_5321<V> item, boolean required) implements TagEntry<V> {

            @Override
            public void collect(final class_2378<V> registry, final Consumer<class_6880<V>> collector) {
                collector.accept(registry.method_40268(this.item));
            }

            @Override
            public void toNode(final ConfigurationNode target) throws SerializationException {
                if (this.required) {
                    ResourceLocationSerializer.toNode(this.item.method_29177(), target);
                } else {
                    ResourceLocationSerializer.toNode(this.item.method_29177(), target.node(ID));
                    target.node(REQUIRED).set(false);
                }
            }

        }

        record Tag<V>(class_6862<V> tagKey, boolean required) implements TagEntry<V> {

            @Override
            public void collect(final class_2378<V> registry, final Consumer<class_6880<V>> collector) {
                for (final class_6880<V> entry : registry.method_40260(this.tagKey)) {
                    collector.accept(entry);
                }
            }

            @Override
            public void toNode(final ConfigurationNode target) throws SerializationException {
                if (this.required) {
                    target.set(TAG_PREFIX + this.tagKey.comp_327());
                } else {
                    target.node(ID).set(TAG_PREFIX + this.tagKey.comp_327());
                    target.node(REQUIRED).set(false);
                }
            }

        }

    }

}
