/*
 * Copyright (c) 2025, Stellardrift and contributors
 * Copyright (c) 2015, Minecrell <https://github.com/Minecrell>
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package ca.stellardrift.gitpatcher.internal;

import java.io.IOException;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.Locale;
import java.util.stream.Stream;
import org.jetbrains.annotations.Nullable;

/**
 * Internal helper methods.
 *
 * <p>No guarantees of stable API.</p>
 */
public class Utils {
    private Utils() {
    }

    /**
     * Capitalize the first character of the provided string.
     *
     * <p>If provided with an empty string, an empty string will be returned.</p>
     *
     * @param input the input
     * @return the capitalized input
     */
    public static String capitalize(final String input) {
        switch (input.length()) {
            case 0: return "";
            case 1: return input.toUpperCase(Locale.ROOT);
            default:
                final StringBuilder ret = new StringBuilder(input.length());
                final int firstCodePoint = input.codePointAt(0);
                ret.appendCodePoint(Character.toUpperCase(firstCodePoint));
                ret.append(input.substring(Character.charCount(firstCodePoint)));
                return ret.toString();
        }
    }

    /**
     * Delete a directory and any subdirectories.
     *
     * @param dir the directory to delete
     * @throws IOException if deletion fails
     */
    public static boolean deleteRecursively(final Path dir) throws IOException {
        if (!Files.isDirectory(dir)) {
            return false;
        }

        Files.walkFileTree(dir, new FileVisitor<>() {
            @Override
            public FileVisitResult preVisitDirectory(final Path dir, final BasicFileAttributes attrs) {
                return FileVisitResult.CONTINUE;
            }

            @Override
            public FileVisitResult visitFile(final Path file, final BasicFileAttributes attrs) throws IOException {
                Files.delete(file);
                return FileVisitResult.CONTINUE;
            }

            @Override
            public FileVisitResult visitFileFailed(final Path file, final IOException exc) throws IOException {
                throw exc;
            }

            @Override
            public FileVisitResult postVisitDirectory(final Path dir, final @Nullable IOException exc) throws IOException {
                Files.delete(dir);
                return FileVisitResult.CONTINUE;
            }
        });
        return true;
    }

    public static boolean isEmptyDir(final Path dir) throws IOException {
        try (final Stream<Path> children = Files.list(dir)) {
            return children.findAny().isEmpty();
        }
    }
}
