/*
 * This file is part of SpongeAPI, licensed under the MIT License (MIT).
 *
 * Copyright (c) SpongePowered <https://www.spongepowered.org>
 * Copyright (c) contributors
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package org.spongepowered.api.event.action;

import org.spongepowered.api.block.BlockSnapshot;
import org.spongepowered.api.entity.living.Living;
import org.spongepowered.api.event.Cancellable;
import org.spongepowered.api.event.Event;
import org.spongepowered.api.world.server.ServerLocation;
import org.spongepowered.math.vector.Vector3d;

/**
 * Called when a {@link Living} enters a bed to sleep in.
 */
public interface SleepingEvent extends Event {

    /**
     * Gets the {@link Living}.
     *
     * @return The humanoid
     */
    Living living();

    /**
     * Gets the {@link BlockSnapshot} of the bed being used to sleep.
     *
     * @return The location of the bed
     */
    BlockSnapshot bed();

    /**
     * Fires before sleeping is attempted.
     */
    interface Pre extends SleepingEvent, Cancellable {}

    /**
     * Fires every tick the {@link Living} is sleeping.
     */
    interface Tick extends SleepingEvent, Cancellable {}

    /**
     * Fires when the {@link Living} tried to sleep but failed.
     */
    interface Failed extends SleepingEvent, Cancellable {

    }

    /**
     * Fires when the {@link Living} wakes up.
     */
    interface Finish extends SleepingEvent {

        /**
         * Gets the previous {@link ServerLocation location} the {@link Living} would have woke at.
         *
         * @return The location
         */
        ServerLocation fromLocation();

        /**
         * Gets the {@link ServerLocation location} the {@link Living} will wake up at.
         *
         * @return The location
         */
        ServerLocation toLocation();

        /**
         * Sets the {@link ServerLocation location} the {@link Living} will wake up at.
         *
         * @param location The location
         */
        void setToLocation(ServerLocation location);

        /**
         * Gets the {@link Vector3d rotation} the {@link Living} would woke up to.
         *
         * @return The rotation
         */
        Vector3d fromRotation();

        /**
         * Gets the {@link Vector3d rotation} the {@link Living} will wake up to.
         *
         * @return The rotation
         */
        Vector3d toRotation();

        /**
         * Sets the {@link Vector3d rotation} the {@link Living} will wake up to.
         *
         * @param rotation The rotation
         */
        void setToRotation(Vector3d rotation);
    }
}
