/*
 * This file is part of SpongeAPI, licensed under the MIT License (MIT).
 *
 * Copyright (c) SpongePowered <https://www.spongepowered.org>
 * Copyright (c) contributors
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package org.spongepowered.api.service.context;

import org.spongepowered.api.event.Cause;

import java.util.Collection;
import java.util.Set;

/**
 * A service that provides shared context calculation functionality for any
 * services that may wish to use contexts.
 *
 * <p>When functionality is provided to query data based upon a given
 * {@link Set} of {@link Context}s, the querying code must consider which of the
 * underlying data is "applicable" to the contexts specified in the query
 * request. It is expected that implementations will deem entries of underlying
 * data to be applicable if the {@link Set} of contexts defined in the query
 * contain all of the {@link Context}s required by the data entry.
 * Mathematically, an underlying data entry is "applicable" if
 * [the context of the entry] is a subset of [the context provided in the
 * query], or programmatically if [the context provided in the query]
 * {@link Set#containsAll(Collection)} of [the context of the entry].</p>
 */
public interface ContextService {

    /**
     * Get the active contexts from the current cause.
     *
     * @return an immutable set of contexts
     */
    Set<Context> contexts();

    /**
     * Extract applicable contexts from the provided {@link Cause}.
     *
     * @param cause source cause
     * @return immutable set of extracted contexts
     */
    Set<Context> contextsFor(Cause cause);

    /**
     * Registers a {@link ContextCalculator} for use by this service.
     *
     * @param calculator The context calculator to register
     */
    void registerContextCalculator(ContextCalculator calculator);

}
